import {ETrainsPartner} from 'projects/trains/constants/trainPartners';

import {ITrainsOrderDetails} from 'reducers/trains/order/types';
import {ITrainsDetailsApiParams} from 'server/api/TrainsApi/types/ITrainsDetailsApiParams';
import {ITrainsDetailsApiResponse} from 'server/api/TrainsApi/types/ITrainsDetailsApiResponse';
import {ITrainsLogBanditApiParams} from 'server/api/TrainsApi/types/ITrainsLogBanditApiParams';
import {EApiEntry} from 'types/EApiEntry';
import ITrainsApiOrderDetails from 'server/api/TrainsApi/types/ITrainsApiOrderDetails';
import {
    ITrainsGetTariffsApiParams,
    ITrainsGetTariffsApiResponse,
} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/ITrainsGetTariffsApi';

import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {RestApiClient} from 'server/utilities/RestApiClient';
import prepareTrainApiOrderStatus from 'server/api/TrainsApi/utilities/prepareTrainApiOrderStatus';

import {IDependencies} from 'server/getContainerConfig';

export class TrainsApi extends RestApiClient {
    private isDirtyRequest: boolean;

    constructor({
        logger,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        requestId,
        rootSpan,
        isBot,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.TRAINS),
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAINS);
        this.isDirtyRequest = isBot;
    }

    activePartners(language: string): Promise<ETrainsPartner[]> {
        return this.get<{partnerCodes: ETrainsPartner[]}>(
            `/${language}/api/active-partners/`,
            {timeout: 1000},
        ).then(data => data.partnerCodes);
    }

    getTariffs({
        language,
        poll,
        ...params
    }: ITrainsGetTariffsApiParams): Promise<ITrainsGetTariffsApiResponse> {
        if (this.isDirtyRequest && !poll) {
            return Promise.resolve({
                segments: [],
                querying: false,
            });
        }

        return this.get<ITrainsGetTariffsApiResponse>(
            `/${language}/api/segments/train-tariffs/${poll ? 'poll/' : ''}`,
            {params},
        );
    }

    trainDetails(
        params: ITrainsDetailsApiParams,
    ): Promise<ITrainsDetailsApiResponse> {
        return this.get<ITrainsDetailsApiResponse>(
            `/ru/api/ufs-train-details/`,
            {
                params: {
                    ...params,
                    enableManyPlacesAtOnce: true,
                    experiment: false,
                    partner: ETrainsPartner.IM,
                    includePriceFee: '1',
                },
            },
        );
    }

    updateOrder(orderId: string): Promise<ITrainsOrderDetails> {
        return this.get<{order: ITrainsApiOrderDetails}>(
            `/ru/api/update-tickets-status/${orderId}/`,
            {
                params: {
                    firstActualWarningOnly: true,
                },
            },
        ).then(({order}) => {
            return {
                ...order,
                status: prepareTrainApiOrderStatus(order.status),
                startStation:
                    order.startStation.popularTitle || order.startStation.title,
                endStation:
                    order.endStation.popularTitle || order.endStation.title,
            };
        });
    }

    /**
     * Для логирования по проекту "Бандит"
     * https://st.yandex-team.ru/TRAVELFRONT-2995
     */
    logBandit(params: ITrainsLogBanditApiParams): Promise<void> {
        return this.post<void>('/ru/api/log-bandit/', params);
    }

    downloadBlank(
        orderId: string,
        blankId?: string,
    ): Promise<NodeJS.ReadableStream> {
        return this.get<NodeJS.ReadableStream>(
            `/ru/api/download-blank/${orderId}/`,
            {
                params: {
                    blank_id: blankId,
                },
                responseType: 'stream',
                maxRedirects: 0,
            },
        );
    }

    protected interceptRequest(request: IRequestConfig): void | Promise<void> {
        request.params = {
            ...request.params,
            _rid: this.requestId,
        };
    }
}
