import snakecaseKeys from 'snakecase-keys';

import {CommonHeaders} from 'server/constants/headers';

import ITrainsChangeRegistrationStatusParams from 'server/api/TrainsBookingApi/types/ITrainsChangeRegistrationStatusParams';
import {
    ITrainsTestContextTokenAnswer,
    ITrainsTestContextTokenParams,
} from 'server/api/TrainsBookingApi/types/ITrainsTestContextToken';
import {EApiEntry} from 'types/EApiEntry';
import ITrainsDownloadBlankApiParams from 'server/api/TrainsBookingApi/types/ITrainsDownloadBlankApiParams';

import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {RestApiClient} from 'server/utilities/RestApiClient';

import {IDependencies} from 'server/getContainerConfig';

/**
 * Оркестровые ручки, для создания заказа поездов.
 *
 * @see https://api.travel-balancer-test.yandex.net/swagger-ui.html#/trains-booking-flow-controller
 */
export class TrainsBookingApi extends RestApiClient {
    private readonly serviceTicket: string;

    constructor({
        logger,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        requestId,
        rootSpan,
    }: IDependencies) {
        super({
            baseURL: getApiHost(EApiEntry.TRAVEL),
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
    }

    /**
     * Добавление страховки в существующую бронь.
     */
    async addInsurance(id: string): Promise<void> {
        return await this.post<void>('/trains_booking_flow/v1/add_insurance', {
            id,
        });
    }

    /**
     * Смена электронной регистрации
     */
    async orderRegistrationStatusChange(
        params: ITrainsChangeRegistrationStatusParams,
    ): Promise<unknown> {
        // @ts-ignore order не соответствует типу для snakecaseKeys()
        const body = snakecaseKeys(params, {deep: true});

        return await this.post<unknown>(
            '/trains_booking_flow/v1/change_registration_status',
            body,
        );
    }

    downloadBlank(
        params: ITrainsDownloadBlankApiParams,
    ): Promise<NodeJS.ReadableStream> {
        return this.get<NodeJS.ReadableStream>(
            '/trains_booking_flow/v1/download_blank',
            {
                params,
                responseType: 'stream',
                maxRedirects: 0,
                headers: {
                    [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
                },
            },
        );
    }

    testContextToken(
        params: ITrainsTestContextTokenParams,
    ): Promise<ITrainsTestContextTokenAnswer> {
        return this.get('/test_context/v1/train_token', {
            params,
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
        };
    }
}
