import {CommonHeaders} from 'server/constants/headers';

import {EApiEntry} from 'types/EApiEntry';
import {
    ITrainsSearchRequestParams,
    ITrainsSearchInfoResponse,
} from './types/ITrainsSearch';
import {
    ITrainsCrossLinksRequestParams,
    ITrainsCrossLinksResponse,
} from 'server/api/TrainsSearchApi/types/ITrainsCrossLinks';
import {
    ITrainsGetDirectionSearchApiParams,
    ITrainsGetDirectionSearchApiResponse,
} from 'server/api/TrainsSearchApi/types/ITrainsGetDirectionSearch';
import {
    ITrainsPriceCalendarRequestParams,
    ITrainsPriceCalendarResponse,
} from './types/ITrainsPriceCalendar';
import {
    ITrainsCanonicalRequestParams,
    ITrainsCanonicalResponse,
} from 'server/api/TrainsSearchApi/types/ITrainsCanonical';

import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {RestApiClient} from 'server/utilities/RestApiClient';

import {IDependencies} from 'server/getContainerConfig';

export class TrainsSearchApi extends RestApiClient {
    private readonly serviceTicket: string;
    private readonly isBot: boolean;

    constructor({
        logger,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        requestId,
        rootSpan,
        isBot,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.TRAVEL)}/trains/v1`,
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.isBot = isBot;
        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
    }

    async search(
        searchParams: ITrainsSearchRequestParams,
    ): Promise<ITrainsSearchInfoResponse> {
        const searchRequestParams: ITrainsSearchRequestParams = {
            ...searchParams,
            isBot: this.isBot || searchParams.isBot,
        };

        return this.get<ITrainsSearchInfoResponse>('/search', {
            params: searchRequestParams,
        });
    }

    async getCanonical(
        params: ITrainsCanonicalRequestParams,
    ): Promise<ITrainsCanonicalResponse> {
        return this.get<ITrainsCanonicalResponse>('/canonical', {
            params,
        });
    }

    async priceCalendar(
        params: ITrainsPriceCalendarRequestParams,
    ): Promise<ITrainsPriceCalendarResponse> {
        const priceCalendarRequestParams: ITrainsPriceCalendarRequestParams = {
            ...params,
            isBot: this.isBot || params.isBot,
        };

        return this.get<ITrainsPriceCalendarResponse>('/price-calendar', {
            params: priceCalendarRequestParams,
        });
    }

    async getCrossLinks(
        params: ITrainsCrossLinksRequestParams,
    ): Promise<ITrainsCrossLinksResponse> {
        return this.get<ITrainsCrossLinksResponse>('/crosslinks', {
            params,
        });
    }

    async getDirectionSearch(
        params: ITrainsGetDirectionSearchApiParams,
    ): Promise<ITrainsGetDirectionSearchApiResponse> {
        return await this.get<ITrainsGetDirectionSearchApiResponse>(
            '/direction',
            {
                params,
            },
        );
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }
}
