import {
    ITravelerDTO,
    ITravelerMutationDTO,
} from 'server/api/TravelersApi/types/ITravelerDTO';
import {
    IPassengerDTO,
    IPassengerMergeDTO,
    IPassengerWithBonusCardsDTO,
    IPassengerWithDocumentsAndBonusCardsDTO,
    IPassengerWithDocumentsDTO,
    TPassengerMutationDTO,
} from 'server/api/TravelersApi/types/IPassengerDTO';
import {
    IDocumentDTO,
    IDocumentMutationDTO,
} from 'server/api/TravelersApi/types/IDocumentDTO';
import {
    IBonusCardDTO,
    IBonusCardMutationDTO,
} from 'server/api/TravelersApi/types/IBonusCardDTO';
import {IDocumentTypesDTO} from 'server/api/TravelersApi/types/IDocumentTypesDTO';
import {EApiEntry} from 'types/EApiEntry';

import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {RestApiClient} from 'server/utilities/RestApiClient';

import {IDependencies} from 'server/getContainerConfig';

/**
 * @deprecated - использовать notebookApi
 */
export class TravelersApi extends RestApiClient {
    private readonly userTicket: string;
    private readonly serviceTicket: string;

    constructor({
        logger,
        requestId,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        blackbox,
        rootSpan,
    }: IDependencies) {
        super({
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            baseURL: getApiHost(EApiEntry.TRAVELLERS),
            rootSpan,
        });

        this.srcParams = getSrcParams(EApiEntry.TRAVELLERS);
        this.userTicket = blackbox.userTicket;
        this.serviceTicket = getServiceTicket('travellersAPI');
    }

    getTraveler(uid: string): Promise<ITravelerDTO | string> {
        return this.get<ITravelerDTO>(`/v1/travelers/${uid}`, {
            validateStatus: status => status === 404 || status === 200,
        });
    }

    createOrUpdateTraveler(
        uid: string,
        traveler: ITravelerMutationDTO,
    ): Promise<ITravelerDTO> {
        return this.post<ITravelerDTO>(`/v1/travelers/${uid}`, traveler);
    }

    getAllPassengers(
        uid: string,
        fields: ['documents', 'bonus-cards'] | ['bonus-cards', 'documents'],
    ): Promise<IPassengerWithDocumentsAndBonusCardsDTO[]>;
    getAllPassengers(
        uid: string,
        fields: ['documents'],
    ): Promise<IPassengerWithDocumentsDTO[]>;
    getAllPassengers(
        uid: string,
        fields: ['bonus-cards'],
    ): Promise<IPassengerWithBonusCardsDTO[]>;
    getAllPassengers(
        uid: string,
        fields:
            | ['documents', 'bonus-cards']
            | ['bonus-cards', 'documents']
            | ['bonus-cards']
            | ['documents'],
    ): Promise<
        | IPassengerWithDocumentsAndBonusCardsDTO[]
        | IPassengerWithBonusCardsDTO[]
        | IPassengerWithDocumentsDTO[]
    >;
    getAllPassengers(uid: string): Promise<IPassengerDTO[]>;
    getAllPassengers(uid: string, fields?: string[]): any {
        return this.get<IPassengerDTO[]>(`/v1/travelers/${uid}/passengers`, {
            params: {fields: (fields || []).join(',')},
        });
    }

    getPassenger(
        uid: string,
        passengerId: string,
        fields: ['documents', 'bonus-cards'] | ['bonus-cards', 'documents'],
    ): Promise<IPassengerWithDocumentsAndBonusCardsDTO>;
    getPassenger(
        uid: string,
        passengerId: string,
        fields: ['documents'],
    ): Promise<IPassengerWithDocumentsDTO>;
    getPassenger(
        uid: string,
        passengerId: string,
        fields: ['bonus-cards'],
    ): Promise<IPassengerWithBonusCardsDTO>;
    getPassenger(uid: string, passengerId: string): Promise<IPassengerDTO>;
    getPassenger(uid: string, passengerId: string, fields?: string[]): any {
        return this.get<IPassengerDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}`,
            {params: {fields: (fields || []).join(',')}},
        );
    }

    createPassenger(
        uid: string,
        passenger: TPassengerMutationDTO,
    ): Promise<IPassengerDTO> {
        return this.post<IPassengerDTO>(
            `/v1/travelers/${uid}/passengers`,
            passenger,
        );
    }

    updatePassenger(
        uid: string,
        passengerId: string,
        passenger: TPassengerMutationDTO,
    ): Promise<IPassengerDTO> {
        return this.put<IPassengerDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}`,
            passenger,
        );
    }

    deletePassenger(uid: string, passengerId: string): Promise<void> {
        return this.delete<void>(
            `/v1/travelers/${uid}/passengers/${passengerId}`,
        );
    }

    getAllDocumentsForPassenger(
        uid: string,
        passengerId: string,
    ): Promise<IDocumentDTO[]> {
        return this.get<IDocumentDTO[]>(
            `/v1/travelers/${uid}/passengers/${passengerId}/documents`,
        );
    }

    getDocument(
        uid: string,
        passengerId: string,
        documentId: string,
    ): Promise<IDocumentDTO> {
        return this.get<IDocumentDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}/documents/${documentId}`,
        );
    }

    createDocument(
        uid: string,
        passengerId: string,
        document: IDocumentMutationDTO,
    ): Promise<IDocumentDTO> {
        return this.post<IDocumentDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}/documents`,
            document,
        );
    }

    updateDocument(
        uid: string,
        passengerId: string,
        documentId: string,
        document: IDocumentMutationDTO,
    ): Promise<IDocumentDTO> {
        return this.put<IDocumentDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}/documents/${documentId}`,
            document,
        );
    }

    deleteDocument(
        uid: string,
        passengerId: string,
        documentId: string,
    ): Promise<void> {
        return this.delete<void>(
            `/v1/travelers/${uid}/passengers/${passengerId}/documents/${documentId}`,
        );
    }

    getAllBonusCardsForPassenger(
        uid: string,
        passengerId: string,
    ): Promise<IBonusCardDTO[]> {
        return this.get<IBonusCardDTO[]>(
            `/v1/travelers/${uid}/passengers/${passengerId}/bonus-cards`,
        );
    }

    getBonusCard(
        uid: string,
        passengerId: string,
        bonusCardId: string,
    ): Promise<IBonusCardDTO> {
        return this.get<IBonusCardDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}/bonus-cards/${bonusCardId}`,
        );
    }

    createBonusCard(
        uid: string,
        passengerId: string,
        bonusCard: IBonusCardMutationDTO,
    ): Promise<IBonusCardDTO> {
        return this.post<IBonusCardDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}/bonus-cards`,
            bonusCard,
        );
    }

    updateBonusCard(
        uid: string,
        passengerId: string,
        bonusCardId: string,
        bonusCard: IBonusCardMutationDTO,
    ): Promise<IBonusCardDTO> {
        return this.put<IBonusCardDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}/bonus-cards/${bonusCardId}`,
            bonusCard,
        );
    }

    deleteBonusCard(
        uid: string,
        passengerId: string,
        bonusCardId: string,
    ): Promise<void> {
        return this.delete<void>(
            `/v1/travelers/${uid}/passengers/${passengerId}/bonus-cards/${bonusCardId}`,
        );
    }

    /**
     * Удаляет всех пассажиров с идентификаторами из списка `passengers`,
     * переносит все их документы и бонусные карты пассажиру
     * с идентификатором `passengerId` и обновляет у него поля, переданные в запросе.
     */
    mergePassengers(
        uid: string,
        passengerId: string,
        passengers: string[],
        updatedFields?: TPassengerMutationDTO,
    ): Promise<IPassengerDTO> {
        return this.post<IPassengerDTO>(
            `/v1/travelers/${uid}/passengers/${passengerId}/combine`,
            {
                passengers,
                ...updatedFields,
            } as IPassengerMergeDTO,
        );
    }

    getDocumentTypes(): Promise<IDocumentTypesDTO> {
        return this.get<IDocumentTypesDTO>('/v1/document_types');
    }

    protected interceptRequest(request: IRequestConfig): void {
        request.timeout = 25000;

        request.headers = {
            ...request.headers,
            'X-Ya-Service-Ticket': this.serviceTicket,
            'X-Ya-User-Ticket': this.userTicket,
        };
    }
}
