import {CommonHeaders} from 'server/constants/headers';
import {
    MOCK_ACTIVE_TRIPS_COUNT,
    MOCK_PAST_TRIPS_COUNT,
    MOCK_NOTIFICATIONS,
    MOCK_RESTRICTIONS,
    MOCK_ACTIVITY_TYPES,
    MOCK_FORECAST,
} from 'server/api/TripsApi/constants/mockCookieName';

import {EApiEntry} from 'types/EApiEntry';
import IGetActiveTripsApiRequest from 'server/api/TripsApi/types/IGetActiveTripsApiRequest';
import IGetActiveTripsApiResponse from 'server/api/TripsApi/types/IGetActiveTripsApiResponse';
import IGetTripsApiRequest from 'server/api/TripsApi/types/IGetTripsApiRequest';
import IGetTripsApiResponse from 'server/api/TripsApi/types/IGetTripsApiResponse';
import IGetMoreTripsApiRequest from 'server/api/TripsApi/types/IGetMoreTripsApiRequest';
import IGetMoreTripsApiResponse from 'server/api/TripsApi/types/IGetMoreTripsApiResponse';
import IGetTripRequestRequest from 'server/api/TripsApi/types/IGetTripRequestRequest';
import IGetTripResponse from 'server/api/TripsApi/types/IGetTripResponse';
import {ICookies} from '@yandex-data-ui/core/lib/types';
import IGetTripAsyncBlocksRequest from 'server/api/TripsApi/types/IGetTripAsyncBlocksRequest';
import IGetTripAsyncBlocksResponse from 'server/api/TripsApi/types/IGetTripAsyncBlocksResponse';
import ETripNotificationType from 'types/trips/ITripNotifications/ETripNotificationType';
import ETripActivityType from 'types/trips/ITripActivities/ETripActivityType';

import {IRequestConfig} from 'server/utilities/HttpClient/IHttpClient';
import {RestApiClient} from 'server/utilities/RestApiClient';
import parseNotificationsFromJSON from 'server/api/TripsApi/utilities/parseNotificationsFromJSON';
import parseActivityTypesFromJSON from 'server/api/TripsApi/utilities/parseActivityTypesFromJSON';
import parseCount from 'server/api/TripsApi/utilities/parseCount';

import {IDependencies} from 'server/getContainerConfig';

import activeTrips from './__mocks__/activeTrips';
import createTripAsyncBlocks from './__mocks__/tripAsyncBlocks';
import getTrips from './__mocks__/trips';
import moreTrips from './__mocks__/moreTrips';
import getTripResponse200 from './__mocks__/tripResponse200';

export class TripsApi extends RestApiClient {
    private readonly serviceTicket: string;
    private readonly cookies: ICookies;
    private readonly mockTrips: boolean;

    constructor({
        logger,
        httpClient,
        sendClickHouseStats,
        getApiHost,
        getSrcParams,
        getServiceTicket,
        requestId,
        rootSpan,
        cookies,
        mockTrips,
    }: IDependencies) {
        super({
            baseURL: `${getApiHost(EApiEntry.TRAVEL)}/trips/v1`,
            logger,
            requestId,
            httpClient,
            sendClickHouseStats,
            rootSpan,
        });

        this.cookies = cookies;
        this.mockTrips = mockTrips;
        this.srcParams = getSrcParams(EApiEntry.TRAVEL);
        this.serviceTicket = getServiceTicket('travelAPI');
    }

    async getActiveTrips(
        params: IGetActiveTripsApiRequest,
    ): Promise<IGetActiveTripsApiResponse> {
        if (this.mockTrips) {
            return {
                trips: activeTrips.slice(0, this.getMockActiveTripsCount()),
            };
        }

        return this.get<IGetActiveTripsApiResponse>('get_active_trips', {
            params,
        });
    }

    async getTrips(params: IGetTripsApiRequest): Promise<IGetTripsApiResponse> {
        if (this.mockTrips) {
            return getTrips({
                activeTripsCount: this.getMockActiveTripsCount(),
                pastTripsCount: this.getMockPastTripsCount(),
                mockNotifications: this.getMockNotifications(),
            });
        }

        return this.get<IGetTripsApiResponse>('get_trips', {params});
    }

    async getMoreTrips(
        params: IGetMoreTripsApiRequest,
    ): Promise<IGetMoreTripsApiResponse> {
        if (this.mockTrips) {
            return moreTrips;
        }

        return this.get<IGetTripsApiResponse>('get_more_trips', {params});
    }

    async getTrip(params: IGetTripRequestRequest): Promise<IGetTripResponse> {
        if (this.mockTrips) {
            return getTripResponse200({
                mockNotifications: this.getMockNotifications(),
                mockRestrictions: this.getMockRestrictions(),
                mockForecast: this.getMockForecast(),
            });
        }

        return this.get<IGetTripResponse>('get_trip', {params});
    }

    async getTripAsyncBlocks(
        params: IGetTripAsyncBlocksRequest,
    ): Promise<IGetTripAsyncBlocksResponse> {
        if (this.mockTrips) {
            return createTripAsyncBlocks({
                mockActivityTypes: this.getMockActivityTypes(),
            });
        }

        return this.get<IGetTripAsyncBlocksResponse>('get_trip_async_blocks', {
            params,
        });
    }

    protected async interceptRequest(request: IRequestConfig): Promise<void> {
        request.headers = {
            ...request.headers,
            [CommonHeaders.X_YA_SERVICE_TICKET]: this.serviceTicket,
            [CommonHeaders.X_YA_USE_CAMELCASE]: 'true',
        };
    }

    private getMockActiveTripsCount(): number | undefined {
        return parseCount(this.cookies[MOCK_ACTIVE_TRIPS_COUNT]);
    }

    private getMockPastTripsCount(): number | undefined {
        return parseCount(this.cookies[MOCK_PAST_TRIPS_COUNT]);
    }

    private getMockNotifications(): ETripNotificationType[] {
        return parseNotificationsFromJSON(this.cookies[MOCK_NOTIFICATIONS]);
    }

    private getMockRestrictions(): boolean {
        return this.cookies[MOCK_RESTRICTIONS] === 'true';
    }

    private getMockActivityTypes(): ETripActivityType[] {
        return parseActivityTypesFromJSON(this.cookies[MOCK_ACTIVITY_TYPES]);
    }

    private getMockForecast(): number | undefined {
        return parseCount(this.cookies[MOCK_FORECAST]);
    }
}
