import moment from 'moment';

import IGetTripResponse from 'server/api/TripsApi/types/IGetTripResponse';
import ETripBlockType from 'types/trips/ETripBlockType';
import ETripOrderState from 'types/trips/ETripOrderState';
import {TTripNotificationsBlock} from 'types/trips/TTripBlock/TTripBlock';
import ETripNotificationType from 'types/trips/ITripNotifications/ETripNotificationType';
import {isNotUndefined} from 'types/utilities';
import {TGetCountryRestrictionsRequest} from 'server/api/AviaCountryRestrictionsApi/types/TAviaCountryRestrictionsApiRequest';
import ETripState from 'types/trips/ETripState';

import {ROBOT} from 'utilities/dateUtils/formats';

import {
    getAviaCheckInNotification,
    getHotelDeferredPaymentNotification,
} from 'server/api/TripsApi/__mocks__/notifications';
import {getForecast} from 'server/api/TripsApi/__mocks__/forecast';

import mockAirlines from './mockAirlines';

interface IGetTripResponse200Params {
    mockNotifications: ETripNotificationType[];
    mockRestrictions: boolean;
    mockForecast: number | undefined;
}

export default function getTripResponse200({
    mockNotifications,
    mockRestrictions,
    mockForecast,
}: IGetTripResponse200Params): IGetTripResponse {
    const mock: IGetTripResponse = {
        trip: {
            id: '1',
            image: 'https://avatars.mds.yandex.net/get-travel-hotels/2456341/region_image_213_0/region-desktop',
            title: 'Санкт-Петербург',
            state: ETripState.CONFIRMED,
            dateForward: '',
            dateBackward: '',
            displayDate: '24 июн — 4 июл, 2021',
            blocks: [
                {
                    type: ETripBlockType.AVIA_ORDER,
                    orders: [
                        {
                            id: '45686023-678f-455c-9b8c-8faafadc9992',
                            title: 'Санкт–Петербург — Александровск–Сахалинск',
                            displayDateForward: '24 ноя, 17:14',
                            displayDateBackward: '24 дек, 22:00',
                            pnr: 'JHG78',
                            airlines: [
                                mockAirlines.aeroflot,
                                mockAirlines.pobeda,
                            ],
                            registrationUrl:
                                'https://www.aeroflot.ru/sb/ckin/app/ru-ru',
                            state: ETripOrderState.CONFIRMED,
                        },
                        {
                            id: '2',
                            title: 'Москва — Сочи',
                            displayDateForward: '24 ноя, 17:14',
                            displayDateBackward: '24 дек, 22:00',
                            pnr: 'RTG45',
                            airlines: [],
                            registrationUrl:
                                'https://www.aeroflot.ru/sb/ckin/app/ru-ru',
                            state: ETripOrderState.CONFIRMED,
                        },
                        {
                            id: '3',
                            title: 'Сочи — Екатеринбург',
                            displayDateForward: '24 ноя, 17:14',
                            pnr: 'OHF88',
                            airlines: [mockAirlines.uralAirlines],
                            registrationUrl:
                                'https://www.aeroflot.ru/sb/ckin/app/ru-ru',
                            state: ETripOrderState.CONFIRMED,
                        },
                        {
                            id: '4',
                            title: 'Санкт–Петербург — Александровск–Сахалинск',
                            displayDateForward: '24 ноя, 17:14',
                            pnr: 'VJA91',
                            airlines: [
                                mockAirlines.aeroflot,
                                mockAirlines.pobeda,
                                mockAirlines.uralAirlines,
                            ],
                            registrationUrl:
                                'https://www.aeroflot.ru/sb/ckin/app/ru-ru',
                            state: ETripOrderState.CANCELLED,
                        },
                        {
                            id: '5',
                            title: 'Санкт–Петербург — Махачкала',
                            displayDateForward: '20 ноя, 17:14',
                            pnr: 'VFA91',
                            airlines: [mockAirlines.aeroflot],
                            registrationUrl:
                                'https://www.aeroflot.ru/sb/ckin/app/ru-ru',
                            state: ETripOrderState.CANCELLED,
                        },
                        {
                            id: '6',
                            title: 'Санкт–Петербург — Самара',
                            displayDateForward: '15 ноя, 17:14',
                            pnr: 'VFA94',
                            airlines: [
                                mockAirlines.aeroflot,
                                mockAirlines.pobeda,
                            ],
                            registrationUrl:
                                'https://www.aeroflot.ru/sb/ckin/app/ru-ru',
                            state: ETripOrderState.CANCELLED,
                        },
                    ],
                },
                {
                    type: ETripBlockType.TRAIN_ORDER,
                    orders: [
                        {
                            id: '616d98aa-c89c-4151-91f9-ac349de68447',
                            title: 'Санкт–Петербург — Александровск–Сахалинск',
                            displayDateForward: '24 ноя, 17:14',
                            displayDateBackward: '24 дек, 22:00',
                            trains: [
                                {
                                    number: '339Г',
                                    description:
                                        'Поезд 339Г Москва — Санкт-Петербург, «Ласточка»',
                                },
                                {
                                    number: '146У',
                                    description:
                                        'Поезд 146У Екатеринбург — Москва',
                                },
                            ],
                            hasTransferWithStationChange: false,
                            state: ETripOrderState.CONFIRMED,
                            refundedTicketsCount: 0,
                        },
                        {
                            id: '6',
                            title: 'Москва — Сочи',
                            displayDateForward: '24 ноя, 17:14',
                            displayDateBackward: '24 дек, 22:00',
                            trains: [
                                {
                                    number: '339Г',
                                    description:
                                        'Поезд 339Г Москва — Санкт-Петербург, «Ласточка»',
                                },
                                {
                                    number: '146У',
                                    description:
                                        'Поезд 146У Екатеринбург — Москва',
                                },
                            ],
                            hasTransferWithStationChange: false,
                            state: ETripOrderState.CONFIRMED,
                            refundedTicketsCount: 0,
                        },
                        {
                            id: '7',
                            title: 'Сочи — Екатеринбург',
                            displayDateForward: '24 ноя, 17:14',
                            trains: [
                                {
                                    number: '339Г',
                                    description:
                                        'Поезд 339Г Москва — Санкт-Петербург, «Ласточка»',
                                },
                            ],
                            hasTransferWithStationChange: false,
                            state: ETripOrderState.CONFIRMED,
                            refundedTicketsCount: 0,
                        },
                        {
                            id: '8',
                            title: 'Екатеринбург — Санкт-Петербург',
                            displayDateForward: '3 мая, 23:59',
                            trains: [
                                {
                                    number: '091И',
                                    description:
                                        'Поезд 091И Екатеринбург — Москва',
                                },
                                {
                                    number: '756А',
                                    description:
                                        'Поезд 756А Москва — Санкт-Петербург, «Сапсан»',
                                },
                            ],
                            hasTransferWithStationChange: true,
                            state: ETripOrderState.CONFIRMED,
                            refundedTicketsCount: 3,
                        },
                        {
                            id: '9',
                            title: 'Сочи — Казань',
                            displayDateForward: '24 ноя, 17:14',
                            trains: [
                                {
                                    number: '339Г',
                                    description:
                                        'Поезд 339Г Москва — Санкт-Петербург, «Ласточка»',
                                },
                                {
                                    number: '146У',
                                    description:
                                        'Поезд 146У Екатеринбург — Москва',
                                },
                            ],
                            hasTransferWithStationChange: true,
                            state: ETripOrderState.CANCELLED,
                            refundedTicketsCount: 4,
                        },
                    ],
                },
                {
                    type: ETripBlockType.HOTEL_ORDER,
                    orders: [
                        {
                            id: '20edf375-3aef-4bc2-a265-015bb84caeb0',
                            title: 'Друзья на Банковском',
                            image: 'https://avatars.mds.yandex.net/get-altay/941278/2a0000016156f0a59a6509c7e500386f7f0b/M',
                            displayCheckinCheckoutDates: '24 ноя — 24 дек',
                            address:
                                'Санкт-Петербург, площадь Александра Невского, 2',
                            coordinates: {lon: 37.85804, lat: 55.549389},
                            state: ETripOrderState.CONFIRMED,
                        },
                        {
                            id: '12',
                            title: 'Апартаменты Chalet With 5 Bedrooms in Cogollos de Guadix, With Wonderful Mountain View, Private Pool and Furnished Terrace - 160 km From the Beach',
                            stars: 5,
                            image: 'https://avatars.mds.yandex.net/get-altay/941278/2a0000016156f0a59a6509c7e500386f7f0b/M',
                            displayCheckinCheckoutDates: '24 ноя — 24 дек',
                            address:
                                'Санкт-Петербург, площадь Александра Невского Таврического геральдического, 2',
                            coordinates: {lon: 60.60647, lat: 56.85587},
                            state: ETripOrderState.CONFIRMED,
                        },
                        {
                            id: '13',
                            title: 'Amsterdam Sunny Top Floor',
                            stars: 2,
                            displayCheckinCheckoutDates: '24 ноя — 24 дек',
                            address:
                                'Санкт-Петербург, площадь Александра Невского, 2',
                            coordinates: {lon: 60.60647, lat: 56.85587},
                            state: ETripOrderState.CANCELLED,
                        },
                        {
                            id: '14',
                            title: 'Апартаменты Chalet',
                            stars: 2,
                            image: 'https://avatars.mds.yandex.net/get-altay/941278/2a0000016156f0a59a6509c7e500386f7f0b/M',
                            displayCheckinCheckoutDates: '28 ноя — 29 дек',
                            address:
                                'Санкт-Петербург, площадь Александра Невского, 2',
                            coordinates: {lon: 60.60647, lat: 56.85587},
                            state: ETripOrderState.CANCELLED,
                        },
                    ],
                },
                {
                    type: ETripBlockType.BUS_ORDER,
                    orders: [
                        {
                            id: '587add2d-aea8-42fe-b908-413cdec8f297',
                            title: 'Екатеринбург — Санкт–Петербург',
                            displayDateForward: '24 ноября, 18:00',
                            description:
                                'Екатеринбург, автостанция Орехово — Санкт–Петербург, автовокзал',
                            carrierName: 'ИП Соломатин Юрий Васильевич',
                            state: ETripOrderState.CONFIRMED,
                            refundedTicketsCount: 0,
                        },
                        {
                            id: '14',
                            title: 'Екатеринбург — Уфа',
                            displayDateForward: '24 сентября, 12:20',
                            description:
                                'Екатеринбург, автостанция Орехово — Уфа, автовокзал',
                            state: ETripOrderState.CONFIRMED,
                            refundedTicketsCount: 3,
                        },
                        {
                            id: '15',
                            title: 'Екатеринбург — Санкт–Петербург',
                            displayDateForward: '24 ноября, 18:20',
                            description:
                                'Екатеринбург, автостанция Орехово — Санкт–Петербург, автовокзал',
                            state: ETripOrderState.CANCELLED,
                            refundedTicketsCount: 4,
                        },
                    ],
                },
                {
                    type: ETripBlockType.HOTEL_CROSS_SALE,
                    block: {
                        title: 'Выберите подходящий отель в Санкт–Петербурге',
                        request: {
                            pointKey: 'c2',
                            totalHotelLimit: 7,
                            checkinDate: moment().add(5, 'day').format(ROBOT),
                            checkoutDate: moment().add(8, 'day').format(ROBOT),
                            adults: 1,
                        },
                    },
                },
                {
                    type: ETripBlockType.ACTIVITIES,
                    isLoaded: false,
                },
                {
                    type: ETripBlockType.FORECAST,
                    block: getForecast(mockForecast),
                },
            ],
        },
    };

    const notificationsBlock = createNotificationsBlock(mockNotifications);

    if (notificationsBlock) {
        mock.trip.blocks.push(notificationsBlock);
    }

    if (mockRestrictions) {
        mock.trip.blocks.push({
            type: ETripBlockType.RESTRICTIONS,
            block: {
                fromCountryTitle: 'Россия',
                toCountryTitle: 'Италия',
                request: {
                    fromPointKey: 'c54',
                    toPointKey: 'l205',
                } as TGetCountryRestrictionsRequest,
            },
        });
    }

    return mock;
}

function createNotificationsBlock(
    mockNotifications: ETripNotificationType[],
): TTripNotificationsBlock | null {
    if (!mockNotifications.length) {
        return null;
    }

    return {
        type: ETripBlockType.NOTIFICATIONS,
        block: {
            notifications: mockNotifications
                .map(notificationType => {
                    switch (notificationType) {
                        case ETripNotificationType.AVIA_CHECK_IN:
                            return getAviaCheckInNotification();
                        case ETripNotificationType.HOTEL_DEFERRED_PAYMENT:
                            return getHotelDeferredPaymentNotification();
                        case ETripNotificationType.TRIP_CREATED:
                        case ETripNotificationType.TRIP_UPDATED:
                        default:
                            return undefined;
                    }
                })
                .filter(isNotUndefined),
        },
    };
}
