import times from 'lodash/times';

import IGetTripsApiResponse from 'server/api/TripsApi/types/IGetTripsApiResponse';
import TTripListItem from 'types/trips/TTripListItem/TTripListItem';
import ETripListItemType from 'types/trips/TTripListItem/types/ETripListItemType';
import isRealTripListItem from 'types/trips/TTripListItem/utilities/isRealTripListItem';
import ETripNotificationType from 'types/trips/ITripNotifications/ETripNotificationType';
import {isNotUndefined} from 'types/utilities';
import ETripState from 'types/trips/ETripState';

const getActiveTrips = (): TTripListItem[] => [
    {
        type: ETripListItemType.REAL,
        id: '1',
        title: 'Санкт–Петербург',
        image: 'https://avatars.mds.yandex.net/get-travel-hotels/2456341/region_image_213_0/region-desktop',
        displayDate: 'Послезавтра',
        state: ETripState.CONFIRMED,
    },
    {
        type: ETripListItemType.REAL,
        id: '2',
        title: 'Франция',
        image: undefined,
        displayDate: '24 — 25 мар, 2021',
        state: ETripState.CONFIRMED,
    },
    {
        type: ETripListItemType.ORDER,
        orderId: '5',
        title: 'Лондон',
        image: undefined,
        displayDate: '5 мая, 2021',
        state: ETripState.CONFIRMED,
    },
];

const pastTrips: TTripListItem[] = [
    {
        type: ETripListItemType.REAL,
        id: '3',
        title: 'Сочи',
        image: 'https://avatars.mds.yandex.net/get-travel-hotels/2409838/region_image_239_1/region-desktop',
        displayDate: '20 дек, 2021 - 11 янв, 2022',
        state: ETripState.CONFIRMED,
    },
    {
        type: ETripListItemType.REAL,
        id: '4',
        title: 'Санкт-Петербург — Александровск-Сахалинский',
        image: undefined,
        displayDate: '24 — 25 мар, 2021',
        state: ETripState.CANCELLED,
    },
];

interface IGetTripsParams {
    activeTripsCount: number | undefined;
    pastTripsCount: number | undefined;
    mockNotifications: ETripNotificationType[];
}

export default function getTrips({
    activeTripsCount,
    pastTripsCount,
    mockNotifications,
}: IGetTripsParams): IGetTripsApiResponse {
    const activeTrips = getActiveTrips();

    const activeRealTrip = activeTrips.find(isRealTripListItem);

    if (activeRealTrip && mockNotifications.length) {
        activeRealTrip.notifications = mockNotifications
            .map(notificationType => {
                return {
                    type: notificationType,
                    isUnread: false,
                };
            })
            .filter(isNotUndefined);
    }

    if (activeTripsCount && activeTripsCount > activeTrips.length) {
        activeTrips.push(
            ...times(activeTripsCount - activeTrips.length, index => {
                const trip = activeTrips[0];

                if (!isRealTripListItem(trip)) {
                    return {
                        ...trip,
                        orderId: `${trip.orderId}-${index}`,
                    };
                }

                return {
                    ...trip,
                    id: `${trip.id}-${index}`,
                };
            }),
        );
    }

    return {
        active: {
            trips: activeTrips.slice(0, activeTripsCount),
            continuationToken: null,
        },
        past: {
            trips: pastTrips.slice(0, pastTripsCount),
            continuationToken: pastTripsCount === undefined ? '111' : null,
        },
    };
}
