import {AxiosError} from 'axios';

import {Request, Response} from '@yandex-data-ui/core/lib/types';

import {authorize} from 'server/utilities/decorators/authorize';
import {logAction} from 'server/utilities/decorators/logAction';
import {getBlackboxDefaultEmail} from 'server/utilities/getBlackboxDefaultEmail';

import {IDependencies} from 'server/getContainerConfig';
import {TravelersApi} from 'server/api/TravelersApi/TravelersApi';
import {AccountsService} from 'server/services/AccountsService/AccountsService';
import {handleResponse} from 'server/controllers/handleResponse';

export class PassengerController {
    private uid?: string;

    private travelersApi: TravelersApi;

    private accountsService: AccountsService;

    constructor({travelersApi, userInfo, accountsService}: IDependencies) {
        this.accountsService = accountsService;
        this.travelersApi = travelersApi;
        this.uid = userInfo && 'uid' in userInfo ? userInfo.uid : undefined;
    }

    @authorize
    @logAction
    travellerGet(_req: Request, res: Response): void {
        this.travelersApi.getTraveler(this.uid!).then(
            data => res.send(data),
            error => handleError(error, res),
        );
    }

    @authorize
    @logAction
    travellerAdd(req: Request, res: Response): void {
        this.travelersApi
            .createOrUpdateTraveler(this.uid!, {
                agree: true,
                email: getBlackboxDefaultEmail(req.blackbox)!,
                phone: '',
            })
            .then(
                data => res.send(data),
                error => handleError(error, res),
            );
    }

    @authorize
    @logAction
    passengersWithDocumentsGet(req: Request, res: Response): void {
        this.accountsService
            .getPassengersWithDocuments()
            .then(handleResponse(res), error => handleError(error, res));
    }

    @authorize
    @logAction
    passengersGet(req: Request, res: Response): void {
        this.accountsService.getPassengersWithDocumentsAndBonusCards().then(
            data => res.send(data),
            error => handleError(error, res),
        );
    }

    @authorize
    @logAction
    passengerAdd(req: Request, res: Response): void {
        this.travelersApi.createPassenger(this.uid!, req.body).then(
            data => res.send(data),
            error => handleError(error, res),
        );
    }

    @authorize
    @logAction
    passengerEdit(req: Request, res: Response): void {
        this.travelersApi
            .updatePassenger(this.uid!, req.body.id, req.body)
            .then(
                data => res.send(data),
                error => handleError(error, res),
            );
    }

    @authorize
    @logAction
    passengerRemove(req: Request, res: Response): void {
        this.travelersApi.deletePassenger(this.uid!, req.body.id).then(
            data => res.send(data),
            error => handleError(error, res),
        );
    }

    @authorize
    @logAction
    documentAdd(req: Request, res: Response): void {
        this.travelersApi
            .createDocument(this.uid!, req.body.passenger_id, req.body)
            .then(
                data => res.send(data),
                error => handleError(error, res),
            );
    }

    @authorize
    @logAction
    documentEdit(req: Request, res: Response): void {
        const {
            id: documentId,
            passenger_id: passengerId,
            ...document
        } = req.body;

        this.travelersApi
            .updateDocument(this.uid!, passengerId, documentId, document)
            .then(
                data => res.send(data),
                error => handleError(error, res),
            );
    }

    @authorize
    @logAction
    documentRemove(req: Request, res: Response): void {
        const {id: documentId, passenger_id: passengerId} = req.body;

        this.travelersApi
            .deleteDocument(this.uid!, passengerId, documentId)
            .then(
                data => res.send(data),
                error => handleError(error, res),
            );
    }

    @authorize
    @logAction
    docTypesGet(req: Request, res: Response): void {
        this.travelersApi.getDocumentTypes().then(
            data => res.send(data),
            error => handleError(error, res),
        );
    }
}

/** @depricated - перевезти на server/controllers/handleError.ts */
function handleError(e: AxiosError, res: Response): void {
    const status = (e.response && e.response.status) || 502;

    res.status(status).send({
        error: e.response
            ? {
                  code: 'TRAVELLERS_ERROR',
                  status: status,
                  statusText: e.response.statusText,
              }
            : e,
    });
}
