import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {IBookingCreateOrderRequest} from 'projects/avia/pages/AviaBooking/types/IBookingCreateOrder';
import {IConfirmationCompleteApiRequest} from 'server/api/AviaBookingApi/types/IConfirmationCompleteApiRequest';

import {logAction} from 'server/utilities/decorators/logAction';
import requiredParams from 'server/utilities/decorators/requiredParams';
import {getAviaPaymentContext} from 'server/utilities/avia/getAviaTestContext';

import {IDependencies} from 'server/getContainerConfig';
import {AviaBookingOrdersApi} from 'server/api/AviaBookingApi/AviaBookingApi';
import {AviaBookingService} from 'server/services/AviaBookingService/AviaBookingService';
import {TestContextService} from 'server/services/TestContextService/TestContextService';
import {handleError} from 'server/controllers/handleError';
import {handleResponse} from 'server/controllers/handleResponse';

export class AviaBookingController {
    private aviaBookingApi: AviaBookingOrdersApi;

    private aviaBookingService: AviaBookingService;
    private testContextService: TestContextService;

    constructor({
        aviaBookingOrdersApi: aviaBookingApi,
        aviaBookingService,
        testContextService,
    }: IDependencies) {
        this.aviaBookingApi = aviaBookingApi;
        this.aviaBookingService = aviaBookingService;
        this.testContextService = testContextService;
    }

    @logAction
    @requiredParams(['token'])
    getVariants(req: Request, res: Response): void {
        this.aviaBookingService
            .getVariants(req.query.token)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    createOrder(req: Request, res: Response): void {
        const formData: IBookingCreateOrderRequest = req.body;
        const paymentTestContextToken =
            formData.paymentTestContextToken || getAviaPaymentContext(req);

        this.aviaBookingService
            .createOrder({
                ...formData,
                paymentTestContextToken,
            })
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requiredParams(['orderId'])
    getCompositeState(req: Request, res: Response): void {
        this.aviaBookingApi
            .getCompositeState(req.query.orderId)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    confirmationComplete(req: Request, res: Response): void {
        const body: IConfirmationCompleteApiRequest = req.body;

        this.aviaBookingApi
            .confirmationComplete(body)
            .then(x => res.json(x), handleError(res));
    }

    @logAction
    @requiredParams(['orderId'])
    initPaymentAsync(req: Request, res: Response): void {
        this.aviaBookingApi
            .initPaymentAsync(req.query.orderId, req.body)
            .then(() => res.send(), handleError(res));
    }

    @logAction
    @requiredParams(['orderId'])
    getOrder(req: Request, res: Response): void {
        const {orderId, source} = req.query;

        this.aviaBookingService
            .getOrder({id: orderId, source})
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    getTestContextToken(req: Request, res: Response): void {
        this.testContextService
            .getAviaTestContextToken(req.query)
            .then(handleResponse(res), handleError(res));
    }
}
