import {get} from 'lodash';

import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {TLogUserSearch} from 'server/loggers/avia/AviaUserSearchLog/types/TLogUserSearch';
import {TAviaLogLandingVisit} from 'server/loggers/avia/AviaLandingVisitLog/types/TAviaLogLandingVisit';

import requestMethod from 'server/utilities/decorators/requestMethod';
import {ILogger} from 'server/utilities/Logger';
import {AVIA_USER_SEARCH_LOG_ERROR_NAME} from '../../loggers/avia/utils/aviaLoggerConstants';
import {getUserSplitByRequest} from '../../providers/experiments/utils/getUserSplitByRequest';
import {getLandingName} from './utilities/getLandingName';

import {IDependencies} from 'server/getContainerConfig';
import {createAviaUserSearchLog} from 'server/loggers/avia/AviaUserSearchLog/aviaUserSearchLog';
import {AviaGeoService} from 'server/services/AviaGeoService/AviaGeoService';
import {
    createAviaVariantsLog,
    IAviaVariantsLoggers,
} from 'server/loggers/avia/AviaVariantsLog/aviaVariantsLog';
import {createAviaLandingLog} from 'server/loggers/avia/AviaLandingVisitLog/aviaLandingVisitLog';
import {GeobaseApi} from 'server/api/GeobaseApi/GeobaseApi';

import {ApiBaseController} from '../ApiBaseController';

export class AviaClientLoggingController extends ApiBaseController {
    private userSearchLog: TLogUserSearch;
    private aviaGeoService: AviaGeoService;
    private logger: ILogger;
    private geobaseApi: GeobaseApi;
    private aviaVariantsLog: IAviaVariantsLoggers;
    private aviaLandingVisitLog: TAviaLogLandingVisit;

    constructor({
        fileLoggerWrapperGetter,
        aviaGeoService,
        geobaseApi,
        logger,
    }: IDependencies) {
        super();

        this.userSearchLog = createAviaUserSearchLog(fileLoggerWrapperGetter);
        this.aviaVariantsLog = createAviaVariantsLog(fileLoggerWrapperGetter);
        this.aviaLandingVisitLog = createAviaLandingLog(
            fileLoggerWrapperGetter,
        );
        this.aviaGeoService = aviaGeoService;
        this.logger = logger;
        this.geobaseApi = geobaseApi;
    }

    @requestMethod('POST')
    userSearch(req: Request, res: Response): void {
        this.sendSuccessResponse(res)(null);

        this.geobaseApi
            .traitsByIp()
            // eslint-disable-next-line camelcase
            .then(data => data?.region_id)
            .then(geoId =>
                Promise.all([
                    getUserSplitByRequest(req),
                    geoId ? this.aviaGeoService.getDataByGeoId(geoId) : null,
                ]),
            )
            .then(([experiments, geoLookUp]) => {
                const testBuckets = experiments
                    ? experiments.boxes
                    : 'Experiemnts not fetched';
                const userCityId = String(
                    get(geoLookUp, ['settlement', 'id'], ''),
                );

                this.userSearchLog(req, testBuckets, userCityId);
            })
            .catch(e =>
                this.logger.logError(AVIA_USER_SEARCH_LOG_ERROR_NAME, e),
            );
    }

    @requestMethod('POST')
    show(req: Request, res: Response): void {
        this.sendSuccessResponse(res)(null);

        this.aviaVariantsLog.logClientShow(req);
    }

    @requestMethod('POST')
    action(req: Request, res: Response): void {
        this.sendSuccessResponse(res)(null);

        this.aviaVariantsLog.logClientAction(req);
    }

    @requestMethod('POST')
    landingVisit(req: Request, res: Response): void {
        this.sendSuccessResponse(res)(null);

        getUserSplitByRequest(req).then(experiments => {
            const ref = req.get('referrer') || '';

            this.aviaLandingVisitLog(req, {
                test_buckets: experiments.testIds,
                landing_name: getLandingName(ref),
                url: ref,
            });
        });
    }
}
