import {Request, Response} from '@yandex-data-ui/core/lib/types';

import {logAction} from 'server/utilities/decorators/logAction';
import requestMethod from 'server/utilities/decorators/requestMethod';

import {IDependencies} from 'server/getContainerConfig';
import {AviaService} from 'server/services/AviaService/AviaService';
import {AviaOrderService} from 'server/services/AviaOrderService/AviaOrderService';

import {handleResponse} from '../handleResponse';
import {handleError} from '../handleError';

export class AviaController {
    private aviaService: AviaService;
    private aviaOrderService: AviaOrderService;

    constructor({aviaService, aviaOrderService}: IDependencies) {
        this.aviaService = aviaService;
        this.aviaOrderService = aviaOrderService;
    }

    @logAction
    @requestMethod('GET')
    getGeoLookup(req: Request, res: Response): void {
        this.aviaService
            .getGeoLookup(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getSuggests(req: Request, res: Response): void {
        this.aviaService
            .getSuggests(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getFlightPage(req: Request, res: Response): void {
        this.aviaService
            .getFlightPage(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getOrder(req: Request, res: Response): void {
        this.aviaOrderService
            .getOrder(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getExtendedCountryRestrictions(req: Request, res: Response): void {
        this.aviaService
            .getExtendedCountryRestrictions(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getSearchToCountry(req: Request, res: Response): void {
        const {
            adults,
            children,
            infants,
            from,
            to,
            dateForward,
            klass,
            dateBackward,
        } = req.query;

        this.aviaService
            .getSearchToCountryPage({
                from,
                to,
                dateForward,
                klass,
                dateBackward,
                passengers: {adults, children, infants},
            })
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getSearchToAnywhere(req: Request, res: Response): void {
        const {
            adults,
            children,
            infants,
            from,
            dateForward,
            klass,
            dateBackward,
        } = req.query;

        this.aviaService
            .getSearchToAnywherePage({
                from,
                dateForward,
                klass,
                dateBackward,
                passengers: {adults, children, infants},
            })
            .then(handleResponse(res), handleError(res));
    }
}
