import {Request, Response} from '@yandex-data-ui/core/lib/types';

import {ILogger} from 'server/utilities/Logger';
import {logAction} from 'server/utilities/decorators/logAction';
import requiredParams from 'server/utilities/decorators/requiredParams';
import {getAviaSearchFormData} from 'server/services/AviaSearchService/utils/getAviaSearchFormData';
import {getPriceIndexSearchInterval} from 'server/api/AviaPriceIndexApi/utils/getPriceIndexSearchInterval';
import {convertSearchFormToPriceIndexParams} from 'server/api/AviaPriceIndexApi/utils/convertSearchFormParamsToPriceIndexParams';
import {convertSearchFiltersToPriceIndexFilters} from 'server/api/AviaPriceIndexApi/utils/convertSearchFiltersToPriceIndexFilters';
import requestMethod from 'server/utilities/decorators/requestMethod';

import {IDependencies} from 'server/getContainerConfig';
import {handleError} from 'server/controllers/handleError';
import {handleResponse} from 'server/controllers/handleResponse';
import {AviaPriceIndexService} from 'server/services/AviaPriceIndexService/AviaPriceIndexService';

export class AviaPriceIndexController {
    private aviaPriceIndexService: AviaPriceIndexService;
    private logger: ILogger;

    constructor({aviaPriceIndexService, logger}: IDependencies) {
        this.aviaPriceIndexService = aviaPriceIndexService;
        this.logger = logger;

        logger.addExtra({api: 'AviaPriceIndexController'});
    }

    @logAction
    dynamics(req: Request, res: Response): void {
        const searchForm = getAviaSearchFormData(req);
        const interval = getPriceIndexSearchInterval(req);

        if (!searchForm || !interval) {
            res.status(400).send(searchForm);
            this.logger.logError('Dynamics controller: Incorrect params');

            return;
        }

        const params = convertSearchFormToPriceIndexParams(searchForm);
        const filters = convertSearchFiltersToPriceIndexFilters(req.body.data);

        this.aviaPriceIndexService
            .dynamics(interval, params, filters)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requiredParams(['fromId', 'toId'])
    calendar(req: Request, res: Response): void {
        this.aviaPriceIndexService
            .calendar(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('POST')
    minPriceBatchSearch(req: Request, res: Response): void {
        const {body} = req;
        const minRequests = body?.minRequests;

        if (
            !minRequests ||
            !Array.isArray(minRequests) ||
            minRequests.length > 5
        ) {
            res.status(400).send();

            return;
        }

        this.aviaPriceIndexService
            .minPriceBatchSearch(body)
            .then(handleResponse(res), handleError(res));
    }
}
