import {URL} from 'url';
import {v4 as uuid} from 'uuid';
import {escape, get, minBy, pick} from 'lodash';

import {
    SOLOMON_AVIA_REDIRECT_ERROR,
    SOLOMON_AVIA_REDIRECT_SUCCESS,
    TRequestPlatform,
} from 'constants/solomon/avia';
import {UTM_QUERY_PARAMS} from 'constants/utm/EUtmQueryParams';

import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {
    IAviaTDClickPrice,
    IAviaTDRedirectAnswer,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDRedirect';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {IUserSplit} from 'server/providers/experiments/types';
import {
    IBaggageTDTariff,
    IVariantPrice,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {IRedirectState} from 'server/services/AviaRedirectService/types/IAviaRedirect';
import {TLogRedirectError} from 'server/loggers/avia/AviaRedirectErrorLog/types/TLogRedirectError';
import {EAviaRedirectErrorReason} from 'server/loggers/avia/AviaRedirectErrorLog/types/EAviaRedirectErrorReason';
import {TLogRedirect} from 'server/loggers/avia/AviaRedirectLog/types/TLogRedirect';
import {IAviaTDRedirectSettings} from 'server/services/AviaRedirectService/types/IAviaTDRedirectSettings';
import {
    TGoalChangingValue,
    TGoalRefundValue,
} from 'utilities/metrika/types/params/avia';
import {EFareFamilyAvailability} from 'server/api/AviaTicketDaemonApi/types/IAviaTDFareFamily';

import {normalizeTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';
import {normalizeTDVariants} from 'reducers/avia/utils/ticketDaemon/normalizeTDVariants';
import {DEFAULT_CURRENCIES} from 'reducers/common/currencies/reducer';

import {denormalizeBaggageInfo} from 'selectors/avia/utils/denormalization/baggage';
import {
    denormalizeRefund,
    getWorstRefund,
} from 'selectors/avia/utils/denormalization/refund';
import {denormalizeChangingCarriage} from 'selectors/avia/utils/denormalization/changingCarriage';
import {processNormalizedVariants} from 'selectors/avia/search/getDenormalizedVariants';

import {logAction} from 'server/utilities/decorators/logAction';
import {track} from 'server/utilities/utm/utm';
import {qidToSearchForm} from 'projects/avia/lib/qid';
import {getSession} from 'server/loggers/avia/AviaVariantsLog/utils/getSession';
import {delay} from 'utilities/async/delay';
import IPrice from 'utilities/currency/PriceInterface';
import {convertPriceToPreferredCurrency} from 'utilities/currency/convertPrice';
import {getNationalVersionForPartnersDisabling} from './utilities/getNationalVersionForPartnersDisabling';
import {getUserSplitByRequest} from 'server/providers/experiments/utils/getUserSplitByRequest';
import {aviaURLs} from 'projects/avia/lib/urls';
import {
    getAviaPaymentContext,
    getAviaTestContext,
} from 'server/utilities/avia/getAviaTestContext';
import {
    getAviaPartnersInfo,
    TVariantPartnersInfo,
} from 'projects/avia/lib/logging/getPartnersInfo';
import AffiliateData from 'server/utilities/DataStorage/AffiliateData/AffiliateData';
import {getWizardFlagsFromQuery} from 'utilities/url/wizard/getWizardFlagsFromQuery';
import {getWizardTariffKeyFromQuery} from 'utilities/url/wizard/getWizardTariffKeyFromQuery';
import {getWizardRedirKeyFromQuery} from 'utilities/url/wizard/getWizardRedirKeyFromQuery';
import {extendOrderData} from './utilities/extendOrderData';

import {handleResponse} from 'server/controllers/handleResponse';
import {handleError} from 'server/controllers/handleError';
import {IDependencies} from 'server/getContainerConfig';
import {AviaRedirectService} from 'server/services/AviaRedirectService/AviaRedirectService';
import {
    createAviaPopularFlightLog,
    TLogPopularFlight,
} from 'server/loggers/avia/AviaPopularFlightLog/aviaPopularFlightLog';
import {
    createAviaRedirectShowLog,
    TAviaRedirectShowLog,
} from 'server/loggers/avia/AviaRedirectShowLog/aviaRedirectShowLog';
import {createAviaRedirectLog} from 'server/loggers/avia/AviaRedirectLog/aviaRedirectLog';
import {CurrenciesService} from 'server/services/CurrenciesService/CurrenciesService';
import {
    createAviaVariantsLog,
    IAviaVariantsLoggers,
} from 'server/loggers/avia/AviaVariantsLog/aviaVariantsLog';
import {createAviaRedirectErrorLog} from 'server/loggers/avia/AviaRedirectErrorLog/aviaRedirectErrorLog';

import {counter as sendSolomonCounter} from '../../../tools/solomon';

type TRedirectDataSettings = Omit<
    IAviaTDRedirectSettings,
    'variantTestContext'
>;

const WAIT_TIMEOUT = 500;
const PAGE_TIMEOUT = 20000;
const STATUS_TIMEOUT = 2000;
const MAX_RETRIES = 10;

const PARTNER_WAIT_TIMEOUT = 300;
const MAX_PARTNER_RETRIES = Math.floor(STATUS_TIMEOUT / PARTNER_WAIT_TIMEOUT);
const WIZARD_REDIR_URL_TIMEOUT = 3000;

const redirectToErrorStatusCodes = [204, 400, 404];

const MAP_AVAILABILITY_TO_LOGS: Record<
    EFareFamilyAvailability,
    TGoalRefundValue
> = {
    [EFareFamilyAvailability.free]: 'free',
    [EFareFamilyAvailability.charge]: 'charged',
    [EFareFamilyAvailability.notAvailable]: 'no',
};

export class AviaRedirectController {
    readonly isFromXredirect: boolean;
    private readonly crowdTestHost: string | undefined;
    private readonly affiliateData: AffiliateData;

    private aviaRedirectService: AviaRedirectService;
    private currenciesService: CurrenciesService;

    private logRedirect: TLogRedirect;
    private logRedirectError: TLogRedirectError;
    private logPopularFlight: TLogPopularFlight;
    private logRedirectShow: TAviaRedirectShowLog;
    private aviaVariantsLog: IAviaVariantsLoggers;

    constructor({
        aviaRedirectService,
        currenciesService,
        fileLoggerWrapperGetter,
        crowdTestHost,
        isFromXredirect,
        affiliateData,
    }: IDependencies) {
        this.aviaRedirectService = aviaRedirectService;
        this.currenciesService = currenciesService;
        this.crowdTestHost = crowdTestHost;
        this.isFromXredirect = isFromXredirect;
        this.affiliateData = affiliateData;

        this.logRedirect = createAviaRedirectLog(fileLoggerWrapperGetter);
        this.logRedirectError = createAviaRedirectErrorLog(
            fileLoggerWrapperGetter,
        );
        this.logPopularFlight = createAviaPopularFlightLog(
            fileLoggerWrapperGetter,
        );
        this.logRedirectShow = createAviaRedirectShowLog(
            fileLoggerWrapperGetter,
        );
        this.aviaVariantsLog = createAviaVariantsLog(fileLoggerWrapperGetter);
    }

    @logAction
    checkVariantExistence(req: Request, res: Response): void {
        this.aviaRedirectService
            .checkVariantExistence(
                req.query,
                getNationalVersionForPartnersDisabling(req),
            )
            .then(handleResponse(res), handleError(res));
    }

    async longRedirect(req: Request, res: Response): Promise<void> {
        const userSplit = await getUserSplitByRequest(req);

        const state = {
            retries: 0,
            statusRetries: 0,
        };

        return this.waitForRedirect(req, PAGE_TIMEOUT, state)
            .then(data =>
                this.redirect(
                    req,
                    res,
                    data,
                    qidToSearchForm(req.query.qid),
                    userSplit,
                ),
            )
            .catch(async e => {
                e.message = `${EAviaRedirectErrorReason.REDIRECT_ERROR}: ${e.message}`;

                this.logRedirectError(req, e, userSplit);

                this.sendSolomonErrorCounter(this.getRequestPlatform(req));

                return res.redirect(
                    aviaURLs.getUrlToRedirectErrorPage(req.query),
                );
            });
    }

    instantRedirect(
        req: Request,
        res: Response,
        experiments: IUserSplit,
    ): Promise<void> {
        const state = {
            retries: 0,
            statusRetries: 0,
            instantRedirect: true,
        };

        return this.waitForRedirect(req, WIZARD_REDIR_URL_TIMEOUT, state).then(
            data =>
                this.redirect(
                    req,
                    res,
                    data,
                    qidToSearchForm(req.query.qid),
                    experiments,
                ),
        );
    }

    private async waitForRedirect(
        req: Request,
        timeout: number,
        state: IRedirectState,
    ): Promise<IAviaTDRedirectAnswer> {
        return new Promise((resolve, reject) => {
            setTimeout(() => {
                reject(new Error('timeout reached'));
            }, timeout);

            const redirectSettings = {ignoreOutdated: !state.instantRedirect};

            this.getVariant(req, redirectSettings, state).then(resolve, reject);
        });
    }

    private getRequestPlatform(req: Request): TRequestPlatform {
        const isMobile = req.uatraits.isMobile;

        return isMobile ? 'mobile' : 'desktop';
    }

    private sendSolomonSuccessCounter(platform: TRequestPlatform): void {
        sendSolomonCounter(SOLOMON_AVIA_REDIRECT_SUCCESS, {platform});
    }

    private sendSolomonErrorCounter(platform: TRequestPlatform): void {
        sendSolomonCounter(SOLOMON_AVIA_REDIRECT_ERROR, {platform});
    }

    private getVariant(
        req: Request,
        redirectSettings: TRedirectDataSettings,
        state: IRedirectState,
    ): Promise<IAviaTDRedirectAnswer> {
        const redirectVariantParams = {
            qid: req.query.qid,
            partner: req.query.partner,
            forward: req.query.forward,
            backward: req.query.backward,
            withBaggage: req.query.withBaggage,
            tariff_sign: req.query.tariff_sign,
            fare_families_hash: req.query.fareFamiliesHash,
            avia_brand: req.query.aviaBrand,
        };
        const redirectWizardParams = {
            wizard_flags: getWizardFlagsFromQuery(req.query),
            wizardTariffKey: getWizardTariffKeyFromQuery(req.query),
            wizard_redir_key: getWizardRedirKeyFromQuery(req.query),
        };
        const redirectData = this.aviaRedirectService.getRedirectData(
            qidToSearchForm(req.query.qid),
            redirectVariantParams,
            redirectWizardParams,
            pick(req.aviaVisitSourceIds, UTM_QUERY_PARAMS),
            {
                ...redirectSettings,
                variantTestContext: getAviaTestContext(req),
            },
        );

        return redirectData.then(
            ({data: {data}, status: statusCode}) => {
                if (redirectToErrorStatusCodes.includes(statusCode)) {
                    throw new Error('No redirect data available');
                }

                if (!data) {
                    return this.retry(req, state);
                }

                const hasVariant = Boolean(data.url);

                if (!hasVariant) {
                    const hasPartnerAnswer =
                        data.partners?.[req.query.partner] === 'done';
                    const newState: IRedirectState = hasPartnerAnswer
                        ? {
                              ...state,
                              statusRetries: state.statusRetries + 1,
                          }
                        : state;

                    return this.retry(req, newState);
                }

                return data;
            },
            e => this.retry(req, state, e),
        );
    }

    private retry(
        req: Request,
        state: IRedirectState,
        error?: unknown,
    ): Promise<IAviaTDRedirectAnswer> {
        const errorStatus = get(error, ['response', 'status']);
        const {disableBoY, instantRedirect} = state;

        const newState = {...state};

        if (!disableBoY && errorStatus) {
            // В случае если тикет-демон ответил статусом 4XX
            // считаем, что BoY не доступен и отправляем пользователя сразу к партнёру
            // указав в параметрах запроса в TD `book_on_yandex = false`
            if (errorStatus >= 400 && errorStatus < 500) {
                newState.disableBoY = true;
            }

            const isLastRetry =
                newState.retries >= MAX_RETRIES ||
                newState.statusRetries >= MAX_PARTNER_RETRIES;

            // если после последнего ретрая (из текущих 10) не получили ссылку на бой - делаем запрос за диплинком на аэрофлот (disable boy) TRAVELFRONT-5794
            if (errorStatus >= 500 && isLastRetry) {
                newState.disableBoY = true;
                newState.retries = MAX_RETRIES - 1;
                newState.statusRetries = MAX_PARTNER_RETRIES - 1;
            }
        }

        const isMaxRetriesExceeded =
            newState.retries >= MAX_RETRIES ||
            newState.statusRetries >= MAX_PARTNER_RETRIES;

        if (isMaxRetriesExceeded) {
            throw new Error('Max retries exceeded');
        }

        newState.retries++;

        const newSettings: TRedirectDataSettings = {
            ignoreOutdated: !instantRedirect,
            book_on_yandex: !newState.disableBoY,
        };

        return delay(WAIT_TIMEOUT)
            .then(() => this.getVariant(req, newSettings, newState))
            .then(data => {
                this.patchRequestWithRedirectMetaInfo(req, newState);

                return data;
            });
    }

    private patchRequestWithRedirectMetaInfo(
        req: Request,
        state: IRedirectState,
    ): void {
        if (state.disableBoY) {
            // Необходимо пометить в логах информацию о том, что
            // редирект на BoY не произошёл и пользователь был отправлен к партнёру
            // Патчим req.aviaVisitSourceIds, так как он пишется в redir-log
            req.aviaVisitSourceIds.eppid = 'portal-no_boy';
        }
    }

    private async writeRedirectLogs(
        req: Request,
        data: IAviaTDRedirectAnswer,
        searchForm: IAviaParams,
        userSplit: IUserSplit,
    ): Promise<void> {
        if (!data.url) {
            return;
        }

        const {nationalVersion, geoId, query} = req;
        const {partner, qid} = query;
        let {variantId} = query;

        const variants = data.variants || {};
        const fares = variants.fares || [];
        const fare = fares[0] || {};
        const prices = fare.prices || [];
        const variant =
            minBy(prices, price => price.tariff.value) || ({} as IVariantPrice);

        const aviaBrand = req.query.aviaBrand;

        const encryptedMarker = data.marker;

        const tariff = variant.tariff || {};
        const currency = geoId
            ? await this.currenciesService.getCurrencyInfo(geoId)
            : DEFAULT_CURRENCIES;
        const rebased = convertPriceToPreferredCurrency({
            ...tariff,
            currenciesInfo: currency,
        });

        const orderData = extendOrderData({
            searchForm,
            data,
            variant,
            qid,
            partner,
            nationalVersion,
            geoId,
        });

        this.logPopularFlight(req, orderData);

        if (!variantId) {
            const session = getSession();
            const actionId = this.aviaVariantsLog.logServerAction(
                req,
                session,
                'redirect',
                userSplit,
            );

            variantId = this.aviaVariantsLog.logServerShow(
                req,
                session,
                actionId,
                orderData,
                getPartnersInfoForLogging(data),
            );
        }

        // Prepare baggage and refund availability
        const normalizedReference = normalizeTDReference(
            data.reference,
            data.partners,
        );

        const {hasBaggage} = denormalizeBaggageInfo(
            variant,
            normalizedReference,
        );

        const variantBaggageAvailability = hasBaggage ? 'yes' : 'no';

        const worstRefundAvailability = getWorstRefund(
            denormalizeRefund(variant, normalizedReference),
        )?.availability;

        const variantRefundAvailability: TGoalRefundValue =
            worstRefundAvailability
                ? MAP_AVAILABILITY_TO_LOGS[worstRefundAvailability]
                : 'unknown';

        const worstChangingCarriageAvailability = denormalizeChangingCarriage(
            variant,
            normalizedReference,
        )?.availability;
        const variantChangedAvailability: TGoalChangingValue =
            worstChangingCarriageAvailability
                ? MAP_AVAILABILITY_TO_LOGS[worstChangingCarriageAvailability]
                : 'unknown';

        this.logRedirectShow(req, orderData);
        this.logRedirect(req, orderData, {
            marker: encryptedMarker,
            url: data.url,
            test_buckets: userSplit.boxes,
            uuid: uuid(),
            user_from_key: null,
            offer_price: Number(rebased.value),
            offer_currency: currency.preferredCurrency,
            original_price: tariff.value,
            original_currency: tariff.currency,
            variantId,
            variantBaggageAvailability,
            variantRefundAvailability,
            variantChangedAvailability,
            avia_brand: aviaBrand,
            ...(data.shown_tariff
                ? {
                      shown_price: convertPriceToPreferredCurrency({
                          value: Number(data.shown_tariff.value),
                          currency: data.shown_tariff.currency,
                          currenciesInfo: currency,
                      }).value,
                      shown_price_unixtime: Number(
                          data.shown_tariff.price_unixtime,
                      ),
                  }
                : {}),
            ...this.affiliateData.getParams(),
        });
    }

    private async redirect(
        req: Request,
        res: Response,
        data: IAviaTDRedirectAnswer,
        searchForm: IAviaParams,
        userSplit: IUserSplit,
    ): Promise<void> {
        if (!data.url) {
            throw new Error('Empty URL. data: ' + JSON.stringify(data));
        }

        await this.writeRedirectLogs(req, data, searchForm, userSplit);

        this.sendSolomonSuccessCounter(this.getRequestPlatform(req));

        // В ответе бекенда приходит захардкоженная ссылка
        // Для тестового окружения подменяем на hostname стенда
        const host = this.crowdTestHost || req.hostname;
        const redirectUrl = this.buildRedirectUrl(
            this.isFromXredirect
                ? data.url
                : data.url.replace('travel-test.yandex.ru', host),
            {
                yaclid: req.query.yaclid,
                paymentTestContext: getAviaPaymentContext(req),
            },
        );

        track(req, res);

        if (data.post) {
            const post = data.post;
            const postRedirect = [
                '<html><body>',
                `<form method="POST" action="${redirectUrl.replace(
                    /"/g,
                    '&quot;',
                )}">`,
                Object.keys(post)
                    .map(key => {
                        return `<input type="hidden" name="${escape(
                            key,
                        )}" value="${escape(post[key])}">`;
                    })
                    .join(''),
                '</form>',
                `<script nonce="${req.nonce}" type="text/javascript">`,
                'window.addEventListener("DOMContentLoaded", function () {',
                'document.forms[0].submit();',
                '});',
                '</script>',
                '</body></html>',
            ].join('');

            res.send(postRedirect);

            return;
        }

        res.redirect(redirectUrl);
    }

    private buildRedirectUrl(
        redirectUrl: string,
        params: {
            yaclid: string | undefined;
            paymentTestContext?: string;
        },
    ): string {
        const url = new URL(redirectUrl);

        if (params.yaclid) {
            url.searchParams.set('yaclid', params.yaclid);
        }

        if (params.paymentTestContext) {
            url.searchParams.set(
                'paymentTestContext',
                params.paymentTestContext,
            );
        }

        return url.toString();
    }
}

/* eslint-disable camelcase */
export interface IAviaOrderData {
    tariff: string;
    partner: string;
    partnerCode: string;
    show_id: string;
    service: string;
    national: string;
    flightsData: IFlightsData;
    qKey: string;
    qid: string;
    routeKey: string;
    baggage: Nullable<IBaggageTDTariff>;
    clickPrice: IAviaTDClickPrice;
    tariffMap: IPrice;
}

interface IFlightsData {
    partner: string;
    from_id: string;
    to_id: string;
    fromStationId: number;
    toStationId: number;
    user_geoid: string;
    forward_numbers: string;
    backward_numbers: string | null;
    date_forward?: string;
    date_backward: string | null;
    forwardRoute: string;
    backwardRoute?: string;
}

function getPartnersInfoForLogging(
    data: IAviaTDRedirectAnswer,
): TVariantPartnersInfo | undefined {
    const normalizedReference = normalizeTDReference(
        data.reference,
        data.partners || {},
    );
    const normalizedVariants = processNormalizedVariants(
        normalizeTDVariants(data.variants),
        normalizedReference,
    );
    const normalizedVariant = minBy(
        normalizedVariants,
        ({price}) => price.tariff.value,
    );

    return normalizedVariant
        ? getAviaPartnersInfo([normalizedVariant], normalizedReference)
        : undefined;
}
