import {Request, Response} from '@yandex-data-ui/core/lib/types';

import {logAction} from 'server/utilities/decorators/logAction';
import requiredParams from 'server/utilities/decorators/requiredParams';
import {getNationalVersionForPartnersDisabling} from './utilities/getNationalVersionForPartnersDisabling';

import {IDependencies} from 'server/getContainerConfig';
import {AviaSearchService} from 'server/services/AviaSearchService/AviaSearchService';
import {AviaPassengerExperienceService} from 'server/services/AviaPassengerExperienceService/AviaPassengerExperienceService';
import {AviaNearService} from 'server/services/AviaNearService/AviaNearService';
import {AviaVariantsService} from 'server/services/AviaVariantsService/AviaVariantsService';
import {AviaOrderService} from 'server/services/AviaOrderService/AviaOrderService';

import {handleError} from '../handleError';
import {handleResponse} from '../handleResponse';

export class AviaSearchController {
    private aviaSearchService: AviaSearchService;
    private aviaPassengerExperienceService: AviaPassengerExperienceService;
    private aviaNearApi: AviaNearService;
    private aviaOrderService: AviaOrderService;
    private aviaVariantsService: AviaVariantsService;

    constructor({
        aviaSearchService,
        aviaPassengerExperienceService,
        aviaNearService,
        aviaVariantsService,
        aviaOrderService,
    }: IDependencies) {
        this.aviaOrderService = aviaOrderService;
        this.aviaSearchService = aviaSearchService;
        this.aviaPassengerExperienceService = aviaPassengerExperienceService;
        this.aviaNearApi = aviaNearService;
        this.aviaVariantsService = aviaVariantsService;
    }

    @logAction
    getPartnersInfo(req: Request, res: Response): void {
        this.aviaSearchService
            .getPartnersInfo()
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    initByQuery(req: Request, res: Response): void {
        this.aviaVariantsService
            .initByQuery(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    results(req: Request, res: Response): void {
        this.aviaVariantsService
            .results(
                {
                    qid: req.query.qid,
                    cont: req.query.cont,
                    allowPortional: req.query.allowPortional,
                },
                getNationalVersionForPartnersDisabling(req),
            )
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    getOrder(req: Request, res: Response): void {
        this.aviaVariantsService
            .order(req.query, getNationalVersionForPartnersDisabling(req))
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    update(req: Request, res: Response): void {
        this.aviaVariantsService
            .results(
                {
                    qid: req.query.qid,
                    cont: req.query.cont,
                    allowPortional: req.query.allowPortional,
                },
                getNationalVersionForPartnersDisabling(req),
            )
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    passengerExperience(req: Request, res: Response): void {
        this.aviaPassengerExperienceService
            .getPassengerExperience(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @requiredParams([
        'fromPointKey',
        'toPointKey',
        'departure',
        'routes',
        'adultSeats',
        'childrenSeats',
        'infantSeats',
    ])
    @logAction
    getGoodPrice(req: Request, res: Response): void {
        this.aviaOrderService
            .getGoodPrice(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @requiredParams(['fromId', 'toId'])
    @logAction
    getPopuplarPartners(req: Request, res: Response): void {
        const {fromId, toId} = req.query;

        this.aviaSearchService
            .getPopularPartners(fromId, toId)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requiredParams(['when', 'fromId', 'toId'])
    topFlights(req: Request, res: Response): void {
        this.aviaSearchService
            .topFlights(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    getNearAirports(req: Request, res: Response): void {
        this.aviaNearApi
            .getNearAirports(req.query)
            .then(handleResponse(res), handleError(res));
    }
}
