import {flatten} from 'lodash';

import {IAviaTDRedirectAnswer} from 'server/api/AviaTicketDaemonApi/types/IAviaTDRedirect';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {
    IVariantPrice,
    IFlight,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {normalizeTDReference} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {
    parseAviaBaggage,
    denormalizePartnersBaggage,
} from 'selectors/avia/utils/denormalization/baggage';

import {decodeQidOrDefault} from 'projects/avia/lib/qid';
import {
    dateStringToIsoDateString,
    dateStringToShortDateString,
} from 'server/controllers/aviaApiContoller/utilities/dateConvertors';
import {getSidBase} from 'server/controllers/aviaApiContoller/utilities/getSidBase';

import {IAviaOrderData} from '../AviaRedirectController';

interface IAviaOrderDataExtenders {
    searchForm: IAviaParams;
    data: IAviaTDRedirectAnswer;
    variant: IVariantPrice;
    qid: string;
    partner: string;
    nationalVersion: string;
    geoId?: number;
}

// TODO: кажется при flight.departure = undefined тут будет что-то невалидное
// https://st.yandex-team.ru/TRAVELFRONT-7155
const flightKey = (flight: IFlight): string =>
    `${flight.number}.${flight.departure?.local.slice(0, -3)}`;

const fligthLogKey = (flight: IFlight): string =>
    `${flight.number}.${dateStringToShortDateString(flight.departure)}`;

export function extendOrderData({
    searchForm,
    data,
    variant,
    qid,
    partner,
    nationalVersion,
    geoId,
}: IAviaOrderDataExtenders): IAviaOrderData {
    const variants = data.variants || {};
    const reference = normalizeTDReference(data.reference, data.partners) || {};
    const fares = variants.fares || [];
    const fare = fares[0] || {};
    const fareRoute = fare.route || [];
    const tariff = variant.tariff || {};
    const qidParams = decodeQidOrDefault(qid);

    const tariffString = tariff.value + ' ' + tariff.currency;

    const flights = reference.flights || {};
    const forwardFlights = fareRoute[0] || [];
    const forwardNumbers = forwardFlights.map(f => flights[f].number).join(';');
    let routeKey = forwardFlights.map(f => fligthLogKey(flights[f])).join('-');
    const forwardRoute = forwardFlights
        .map(f => flightKey(flights[f]))
        .join(',');

    let dateForward;
    const firstForward = flights[forwardFlights[0]] || {};
    const lastForward = flights[forwardFlights[forwardFlights.length - 1]];

    if (firstForward && firstForward.departure) {
        dateForward = dateStringToIsoDateString(firstForward.departure);
    }

    let backwardNumbers;
    let dateBackward;

    const backwardFlights = fareRoute[1] || [];
    let backwardRoute;

    if (backwardFlights.length) {
        backwardRoute = backwardFlights
            .map(f => flightKey(flights[f]))
            .join(',');
        backwardNumbers = backwardFlights.map(f => flights[f].number).join(';');

        const firstBackward = flights[backwardFlights[0]];

        if (firstBackward && firstBackward.departure) {
            dateBackward = dateStringToIsoDateString(firstBackward.departure);
        }

        routeKey +=
            '_' + backwardFlights.map(f => fligthLogKey(flights[f])).join('-');
    }

    const baggage = parseAviaBaggage(
        flatten(denormalizePartnersBaggage(variant, reference.baggageTariffs)),
    );

    return {
        tariff: tariffString,
        partner,
        partnerCode: variant.partnerCode,
        show_id: getSidBase('0'),
        service: variant.service,
        national: nationalVersion,
        flightsData: {
            partner,
            from_id: String(searchForm.fromId),
            to_id: String(searchForm.toId),
            fromStationId: firstForward.from,
            toStationId: lastForward.to,
            user_geoid: String(geoId),
            forward_numbers: forwardNumbers,
            backward_numbers: backwardNumbers || null,
            date_forward: dateForward,
            date_backward: dateBackward || null,
            forwardRoute,
            backwardRoute,
        },
        qKey: qidParams.queryKey,
        qid,
        routeKey,
        baggage,
        clickPrice: data.click_price,
        tariffMap: tariff,
    };
}
