import {Request, Response} from '@yandex-data-ui/core/lib/types';

import {logAction} from 'server/utilities/decorators/logAction';
import requestMethod from 'server/utilities/decorators/requestMethod';
import requiredParams from 'server/utilities/decorators/requiredParams';
import {
    EValidateType,
    validateParams,
} from 'server/utilities/decorators/validateParams';

import {IDependencies} from 'server/getContainerConfig';
import {handleResponse} from 'server/controllers/handleResponse';
import {handleError} from 'server/controllers/handleError';
import {GenericOrderService} from 'server/services/GenericOrderService/GenericOrderService';
import {handlePdfFileStream} from 'server/controllers/handlePdfFileStream';

export class GenericOrderApiController {
    private readonly genericOrderService: GenericOrderService;

    constructor({genericOrderService}: IDependencies) {
        this.genericOrderService = genericOrderService;
    }

    @logAction
    @requestMethod('POST')
    createOrder(req: Request, res: Response): void {
        this.genericOrderService
            .createOrder(req.body, req)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('POST')
    addService(req: Request, res: Response): void {
        this.genericOrderService
            .addService(req.body)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('POST')
    calculateRefundAmount(req: Request, res: Response): void {
        this.genericOrderService
            .calculateRefundAmount(req.body)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('POST')
    cancelOrder(req: Request, res: Response): void {
        this.genericOrderService
            .cancelOrder(req.body)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('POST')
    estimateDiscount(req: Request, res: Response): void {
        this.genericOrderService
            .estimateDiscount(req.body)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getOrder(req: Request, res: Response): void {
        this.genericOrderService
            .getOrder(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    getOrderState(req: Request, res: Response): void {
        this.genericOrderService
            .getOrderState(req.query)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('POST')
    startPayment(req: Request, res: Response): void {
        this.genericOrderService
            .startPayment(req.body)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('POST')
    startRefund(req: Request, res: Response): void {
        this.genericOrderService
            .startRefund(req.body)
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requiredParams(['orderId', 'token'])
    @validateParams([
        {
            name: 'orderId',
            type: EValidateType.CHAR_AND_DIGIT,
        },
        {
            name: 'blankId',
            type: EValidateType.CHAR_AND_DIGIT,
        },
    ])
    downloadBlank(req: Request, res: Response): void {
        const {
            query,
            query: {orderId, blankId},
        } = req;

        this.genericOrderService.downloadBlank(query).then(
            handlePdfFileStream(res, {
                fileName: [orderId, blankId].filter(s => s).join('-'),
            }),
            handleError(res),
        );
    }
}
