import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {
    ILegacyHotelInfoResponse,
    ILegacyHotelInfoError,
} from 'server/api/HotelSearchAPI/types/ILegacyHotelInfo';
import {
    IHotelInfo,
    IHotelInfoError,
} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import {
    IHotelImages,
    IHotelImagesError,
} from 'server/api/HotelSearchAPI/types/IHotelImages';
import {
    IHotelReviewsInfo,
    IHotelReviewsInfoError,
} from 'server/api/HotelSearchAPI/types/IHotelReviews';
import {
    IHotelOffersInfo,
    IHotelOffersInfoError,
} from 'server/api/HotelSearchAPI/types/IOffersInfo';
import {
    ISimilarHotelsInfo,
    ISimilarHotelsInfoError,
} from 'server/api/HotelSearchAPI/types/ISimilarHotels';
import {IHotelReviewReactionError} from 'server/api/HotelSearchAPI/types/IHotelReviewReaction';
import {IAddHotelReviewResponse} from 'server/api/HotelSearchAPI/types/IAddHotelReview';
import {IEditHotelReviewResponse} from 'server/api/HotelSearchAPI/types/IEditHotelReview';
import {IDeleteHotelReviewResponse} from 'server/api/HotelSearchAPI/types/IDeleteHotelReview';
import {IUploadHotelReviewImageResponse} from 'server/api/HotelSearchAPI/types/IUploadHotelReviewImage';
import {IDeleteHotelReviewImagesResponse} from 'server/api/HotelSearchAPI/types/IDeleteHotelReviewImages';
import {ICalendarPricesResponse} from 'server/api/HotelSearchAPI/types/ICalendarPrices';

import requiredParams from 'server/utilities/decorators/requiredParams';
import requestMethod from 'server/utilities/decorators/requestMethod';
import {logAction} from 'server/utilities/decorators/logAction';

import {HotelSearchService} from 'server/services/HotelsServices/HotelSearchService';

import {IDependencies} from '../../getContainerConfig';
import HotelsBaseController, {IBaseErrorResponse} from './HotelsBaseController';

export class HotelSearchController extends HotelsBaseController {
    private hotelSearchService: HotelSearchService;

    constructor({hotelSearchService}: IDependencies) {
        super();

        this.hotelSearchService = hotelSearchService;
    }

    @logAction
    @requestMethod('GET')
    @requiredParams(['id'])
    getLegacyHotelInfo(req: Request, res: Response): void {
        this.hotelSearchService
            .getLegacyHotelInfo(req.query)
            .then(this.sendSuccessResponse<ILegacyHotelInfoResponse>(res))
            .catch(
                this.sendErrorResponse<ILegacyHotelInfoError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('GET')
    getHotelInfo(req: Request, res: Response): void {
        this.hotelSearchService
            .getHotelInfo(req.query)
            .then(this.sendSuccessResponse<IHotelInfo>(res))
            .catch(
                this.sendErrorResponse<IHotelInfoError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('GET')
    getCalendarPrices(req: Request, res: Response): void {
        this.hotelSearchService
            .getCalendarPrices(req.query)
            .then(this.sendSuccessResponse<ICalendarPricesResponse>(res))
            .catch(
                this.sendErrorResponse<IHotelInfoError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('GET')
    @requiredParams(['hotelSlug'])
    getHotelImages(req: Request, res: Response): void {
        this.hotelSearchService
            .getHotelImages(req.query)
            .then(this.sendSuccessResponse<IHotelImages>(res))
            .catch(
                this.sendErrorResponse<IHotelImagesError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('GET')
    @requiredParams(['hotelSlug'])
    getHotelReviews(req: Request, res: Response): void {
        this.hotelSearchService
            .getHotelReviews(req.query)
            .then(this.sendSuccessResponse<IHotelReviewsInfo>(res))
            .catch(
                this.sendErrorResponse<IHotelReviewsInfoError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('POST')
    addHotelReview(req: Request, res: Response): void {
        this.hotelSearchService
            .addHotelReview(req.body)
            .then(this.sendSuccessResponse<IAddHotelReviewResponse>(res))
            .catch(
                this.sendErrorResponse<IBaseErrorResponse>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('POST')
    editHotelReview(req: Request, res: Response): void {
        this.hotelSearchService
            .editHotelReview(req.body)
            .then(this.sendSuccessResponse<IEditHotelReviewResponse>(res))
            .catch(
                this.sendErrorResponse<IBaseErrorResponse>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('POST')
    deleteHotelReview(req: Request, res: Response): void {
        this.hotelSearchService
            .deleteHotelReview(req.body)
            .then(this.sendSuccessResponse<IDeleteHotelReviewResponse>(res))
            .catch(
                this.sendErrorResponse<IBaseErrorResponse>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('POST')
    uploadHotelReviewImage(req: Request, res: Response): void {
        this.hotelSearchService
            .proxyUploadHotelReviewImage(req)
            .then(
                this.sendSuccessResponse<IUploadHotelReviewImageResponse>(res),
            )
            .catch(
                this.sendErrorResponse<IBaseErrorResponse>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('POST')
    deleteHotelReviewImages(req: Request, res: Response): void {
        this.hotelSearchService
            .deleteHotelReviewImages(req.query, req.body)
            .then(
                this.sendSuccessResponse<IDeleteHotelReviewImagesResponse>(res),
            )
            .catch(
                this.sendErrorResponse<IBaseErrorResponse>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('GET')
    @requiredParams(['hotelSlug'])
    getHotelOffers(req: Request, res: Response): void {
        this.hotelSearchService
            .getHotelOffers(req.query)
            .then(this.sendSuccessResponse<IHotelOffersInfo>(res))
            .catch(
                this.sendErrorResponse<IHotelOffersInfoError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('GET')
    @requiredParams(['hotelSlug'])
    getSimilarHotels(req: Request, res: Response): void {
        this.hotelSearchService
            .getSimilarHotels(req.query)
            .then(this.sendSuccessResponse<ISimilarHotelsInfo>(res))
            .catch(
                this.sendErrorResponse<ISimilarHotelsInfoError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    @logAction
    @requestMethod('POST')
    @requiredParams(['hotelSlug', 'reviewId', 'userReaction'])
    setHotelReviewReaction(req: Request, res: Response): void {
        this.hotelSearchService
            .setHotelReviewReaction(req.query)
            .then(this.sendSuccessResponse(res))
            .catch(
                this.sendErrorResponse<IHotelReviewReactionError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }
}
