import {
    IBookOffer,
    IBookOfferInfoError,
} from 'server/api/HotelsBookAPI/types/IBookOffer';
import {IStartPaymentParams} from 'server/api/HotelsBookAPI/types/TStartPayment';
import {IOrderError} from 'server/api/HotelsBookAPI/types/IOrder';
import {
    IOrderRefundableInfo,
    IOrderRefundableInfoError,
} from 'server/api/HotelsBookAPI/types/IOrderRefundableInfo';
import {
    TCancelOrderType,
    ICancelOrderError,
} from 'server/api/HotelsBookAPI/types/ICancelOrder';
import {
    IDownloadVoucherParams,
    EDownloadActionTypes,
    ESearchByOrderId,
} from 'server/api/HotelsBookAPI/types/IDownloadVoucher';
import {ICheckVoucherError} from 'server/api/HotelsBookAPI/types/ICheckVoucher';
import {
    ITestBookOfferTokenResponse,
    ITestBookOfferTokenError,
} from 'server/api/HotelsBookAPI/types/ITestBookOfferToken';
import ICustomerUser from 'server/api/HotelsBookAPI/types/ICustomerUser';
import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {
    ICreateOrderRequest,
    ICreateOrder,
    ICreateOrderError,
} from 'types/hotels/book/ICreateOrder';
import {IApplyPromoCodesRequest} from 'types/hotels/book/IApplyPromoCodes';
import {IRemoveOrderError} from 'server/api/HotelsBookAPI/types/IRemoveOrder';

import {logAction} from 'server/utilities/decorators/logAction';
/* Error helpers */
import {prepareOfferInfoError} from 'server/api/HotelsBookAPI/utilities/prepareOfferInfo';
import {prepareCreateOrderError} from 'server/api/HotelsBookAPI/utilities/prepareCreateOrder';
import {prepareOrderError} from 'server/api/HotelsBookAPI/utilities/prepareOrder';
import {prepareOrderRefundableInfoError} from 'server/api/HotelsBookAPI/utilities/prepareOrderRefundableInfo';
import {prepareCancelOrderError} from 'server/api/HotelsBookAPI/utilities/prepareCancelOrder';
import {prepareCheckVoucherError} from 'server/api/HotelsBookAPI/utilities/prepareCheckVoucher';
import {prepareDownloadVoucherError} from 'server/api/HotelsBookAPI/utilities/prepareDownloadVoucher';
import requestMethod from 'server/utilities/decorators/requestMethod';

import {
    HotelsBookService,
    IDownloadPdf,
} from 'server/services/HotelsServices/HotelsBookService';
import {IDependencies} from 'server/getContainerConfig';
import HotelsBaseController from 'server/controllers/hotelsApiController/HotelsBaseController';

import {handleResponse} from '../handleResponse';
import {handleError} from '../handleError';

export class HotelsBookController extends HotelsBaseController {
    static getCustomerUser(req: Request): ICustomerUser {
        const {
            headers: {'x-real-ip': customerIp, 'user-agent': customerUserAgent},
            userInfo,
            cookies: {yandexuid: customerYandexUid},
        } = req;

        const authUserInfo =
            userInfo && 'isAuth' in userInfo
                ? {
                      customerPassportId: userInfo.uid,
                      customerLogin: userInfo.login,
                  }
                : {};

        return {
            customerIp,
            customerUserAgent,
            customerYandexUid,
            ...authUserInfo,
        };
    }

    private hotelsBookService: HotelsBookService;

    constructor({hotelsBookService}: IDependencies) {
        super();

        this.hotelsBookService = hotelsBookService;
    }

    /* Fetch offer flow */

    @logAction
    offerInfoByToken(req: Request, res: Response): void {
        const {token, label} = req.query;
        const {customerIp, customerUserAgent} =
            HotelsBookController.getCustomerUser(req);

        this.hotelsBookService
            .fetchOfferInfo({
                token,
                label,
                customerIp,
                customerUserAgent,
            })
            .then(handleResponse<IBookOffer>(res))
            .catch(
                this.sendErrorResponse<IBookOfferInfoError>(
                    res,
                    prepareOfferInfoError,
                ),
            );
    }

    /* Create order flow */

    @logAction
    @requestMethod('POST')
    createOrder(req: Request, res: Response): void {
        const createOrderForm: ICreateOrderRequest = req.body;

        this.hotelsBookService
            .createOrder({
                ...HotelsBookController.getCustomerUser(req),
                ...createOrderForm,
            })
            .then(this.sendSuccessResponse<ICreateOrder>(res))
            .catch(
                this.sendErrorResponse<ICreateOrderError>(
                    res,
                    prepareCreateOrderError,
                ),
            );
    }

    @logAction
    @requestMethod('POST')
    estimateDiscount(req: Request, res: Response): void {
        const applyPromoCodes: IApplyPromoCodesRequest = req.body;

        this.hotelsBookService
            .estimateDiscount({
                ...HotelsBookController.getCustomerUser(req),
                ...applyPromoCodes,
            })
            .then(handleResponse(res))
            .catch(handleError(res));
    }

    @logAction
    startPayment(req: Request, res: Response): void {
        const startPaymentParams: IStartPaymentParams = req.body;

        this.hotelsBookService
            .startPayment(startPaymentParams)
            .then(handleResponse(res))
            .catch(handleError(res));
    }

    @logAction
    getOrder(req: Request, res: Response): void {
        const orderParams = req.query;

        this.hotelsBookService
            .getOrder(orderParams)
            .then(handleResponse(res))
            .catch(this.sendErrorResponse<IOrderError>(res, prepareOrderError));
    }

    @logAction
    getOrderStatus(req: Request, res: Response): void {
        const orderParams = req.query;

        this.hotelsBookService
            .getOrderStatus(orderParams)
            .then(handleResponse(res))
            .catch(this.sendErrorResponse<IOrderError>(res, prepareOrderError));
    }

    /* Cancel order flow */

    @logAction
    getOrderRefundableInfo(req: Request, res: Response): void {
        const orderParams = req.query;

        this.hotelsBookService
            .getOrderRefundableInfo(orderParams)
            .then(this.sendSuccessResponse<IOrderRefundableInfo | null>(res))
            .catch(
                this.sendErrorResponse<IOrderRefundableInfoError>(
                    res,
                    prepareOrderRefundableInfoError,
                ),
            );
    }

    @logAction
    cancelOrder(req: Request, res: Response): void {
        const cancelOrderParams = req.body;

        this.hotelsBookService
            .cancelOrder(cancelOrderParams)
            .then(this.sendSuccessResponse<TCancelOrderType>(res))
            .catch(
                this.sendErrorResponse<ICancelOrderError>(
                    res,
                    prepareCancelOrderError,
                ),
            );
    }

    /* Remove order */

    @logAction
    @requestMethod('DELETE')
    removeOrder(req: Request, res: Response): void {
        const {id} = req.params;

        this.hotelsBookService
            .removeOrder(id)
            .then(this.sendSuccessResponse<void>(res))
            .catch(
                this.sendErrorResponse<IRemoveOrderError>(
                    res,
                    prepareCancelOrderError,
                ),
            );
    }

    /* Download business trip doc flow */

    @logAction
    generateBusinessTripPdf(req: Request, res: Response): void {
        this.hotelsBookService
            .generateBusinessTripPdf(req.query)
            .then(handleResponse(res))
            .catch(handleError(res));
    }

    @logAction
    checkBusinessTripPdf(req: Request, res: Response): void {
        this.hotelsBookService
            .checkBusinessTripPdf(req.query)
            .then(handleResponse(res))
            .catch(handleError(res));
    }

    @logAction
    downloadBusinessTripPdf(req: Request, res: Response): void {
        const {voucherFileId: id} = req.params;
        const {actionType = EDownloadActionTypes.DOWNLOAD} = req.query;

        this.hotelsBookService.downloadBusinessTripPdf({id}).then(
            this.sendSuccessFileResponse(res, {
                actionType,
            }),
        );
    }

    /* Download voucher flow */

    @logAction
    checkVoucher(req: Request, res: Response): void {
        const voucherParams = req.query;

        this.hotelsBookService
            .checkVoucher(voucherParams)
            .then(handleResponse(res))
            .catch(
                this.sendErrorResponse<ICheckVoucherError>(
                    res,
                    prepareCheckVoucherError,
                ),
            );
    }

    @logAction
    downloadVoucher(req: Request, res: Response): void {
        const {voucherFileId: orderIdOrPrettyOrderId} = req.params;
        const {actionType, searchBy = ESearchByOrderId.PRETTY_ORDER_ID} =
            req.query;
        const downloadVoucherParams: IDownloadVoucherParams = {
            orderIdOrPrettyOrderId,
            actionType,
            searchBy,
        };

        this.hotelsBookService
            .downloadVoucher(downloadVoucherParams)
            .then(this.sendSuccessFileResponse(res, downloadVoucherParams))
            .catch((apiError: unknown) => {
                const {redirectPath} = prepareDownloadVoucherError(
                    apiError,
                    req.url,
                );

                res.redirect(redirectPath);
            });
    }

    /* Test context flow */

    @logAction
    @requestMethod('GET')
    getTestBookOfferToken(req: Request, res: Response): void {
        this.hotelsBookService
            .getHotelsTestContextToken(req.query)
            .then(this.sendSuccessResponse<ITestBookOfferTokenResponse>(res))
            .catch(
                this.sendErrorResponse<ITestBookOfferTokenError>(
                    res,
                    this.baseApiErrorHandler,
                ),
            );
    }

    private sendSuccessFileResponse =
        (res: Response, {actionType}: {actionType: EDownloadActionTypes}) =>
        ({fileName, pdf}: IDownloadPdf): void => {
            res.type('application/pdf');

            if (actionType === EDownloadActionTypes.DOWNLOAD) {
                res.set(
                    'Content-Disposition',
                    `attachment; filename=${fileName}`,
                );
            }

            pdf.pipe(res);
        };
}
