import {URLs} from 'constants/urls';
import {LANG} from 'constants/langCodes';
import {
    HTTP_BAD_REQUEST_CODE,
    HTTP_INTERNAL_ERROR_CODE,
} from 'server/constants/httpCodes';

import {EFirmTrain} from 'types/trains/seoPages/firmTrain/TFirmTrainBlock';
import {Request, Response} from '@yandex-data-ui/core/lib/types';

import {internalUrl} from 'utilities/url';
import {logAction} from 'server/utilities/decorators/logAction';
import makeProxyInvoker from 'server/utilities/makeProxyInvoker';
import requiredParams from 'server/utilities/decorators/requiredParams';
import getOrderError from 'server/controllers/trainsApiController/utilities/handleOrderError';
import {isHttpClientError} from 'server/utilities/HttpClient/isHttpClientError';
import requestMethod from 'server/utilities/decorators/requestMethod';
import {
    EValidateType,
    validateParams,
} from 'server/utilities/decorators/validateParams';
import {isBadRequestError} from 'server/utilities/HttpClient/isBadRequestError';

import {IDependencies} from 'server/getContainerConfig';
import {TrainsService} from 'server/services/TrainsService/TrainsService';
import {RaspService} from 'server/services/RaspService/RaspService';
import {handleResponse} from 'server/controllers/handleResponse';
import {handlePdfFileStream} from 'server/controllers/handlePdfFileStream';
import {TrainsSearchService} from 'server/services/TrainsSearchService/TrainsSearchService';

export class TrainsApiController {
    private raspService: RaspService;
    private trainsService: TrainsService;
    private trainsSearchService: TrainsSearchService;

    constructor({
        raspService,
        trainsService,
        trainsSearchService,
    }: IDependencies) {
        this.raspService = raspService;
        this.trainsService = trainsService;
        this.trainsSearchService = trainsSearchService;
    }

    @logAction
    getSuggests(req: Request, res: Response): void {
        this.trainsService
            .getSuggests(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @validateParams([
        {
            name: 'language',
            type: EValidateType.ENUM,
            data: {...LANG, undefined},
        },
    ])
    activePartners(req: Request, res: Response): void {
        this.trainsService
            .activePartners(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @validateParams([
        {
            name: 'language',
            type: EValidateType.ENUM,
            data: {...LANG, undefined},
        },
    ])
    getTariffs(req: Request, res: Response): void {
        this.trainsService
            .getTariffs(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    trainDetails(req: Request, res: Response): void {
        this.trainsService
            .getTrainDetails(req.body)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    saveOffer(req: Request, res: Response): void {
        this.trainsService
            .saveOffer(req.body, req)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    saveDocumentsAndTravelerToNotebook(req: Request, res: Response): void {
        this.trainsService
            .saveDocumentsAndTravelerToNotebook(req.body)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    addInsurance(req: Request, res: Response): void {
        this.trainsService
            .addInsurance(req.body)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    parseContext(req: Request, res: Response): void {
        this.raspService
            .parseContext(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('GET')
    @requiredParams(['pointFrom', 'pointTo'])
    priceCalendar(req: Request, res: Response): void {
        this.trainsSearchService
            .priceCalendar(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('GET')
    genericSearch(req: Request, res: Response): void {
        this.trainsSearchService
            .search(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('GET')
    @requiredParams(['fromKey', 'toKey'])
    getCrossLinks(req: Request, res: Response): void {
        this.trainsSearchService
            .getCrossLinks(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    orderSegment(req: Request, res: Response): void {
        const {pointFrom, pointTo, departure, number} = req.query;

        this.raspService
            .getTrainOrderSegment({pointFrom, pointTo, departure, number})
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    countries(req: Request, res: Response): void {
        this.raspService
            .getCountries()
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('POST')
    logBandit(req: Request, res: Response): void {
        this.trainsService
            .logBandit(req.body)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('GET')
    @validateParams([
        {
            name: 'orderId',
            type: EValidateType.CHAR_AND_DIGIT,
        },
    ])
    getActualizedTrainApiOrder(req: Request, res: Response): void {
        this.trainsService
            .getActualizedTrainApiOrder(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('POST')
    changeOrderRegistrationStatus(req: Request, res: Response): void {
        this.trainsService
            .changeOrderRegistrationStatus(req.body)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requiredParams(['id'])
    @validateParams([
        {
            name: 'id',
            type: EValidateType.CHAR_AND_DIGIT,
        },
        {
            name: 'blankId',
            type: EValidateType.CHAR_AND_DIGIT,
        },
    ])
    downloadBlank(req: Request, res: Response): void {
        const {
            query,
            query: {id, blankId, refundOperationId},
        } = req;

        this.trainsService.downloadBlank(query).then(
            handlePdfFileStream(res, {
                fileName: [id, blankId, refundOperationId]
                    .filter(s => s)
                    .join('-'),
            }),
            handleError(req, res),
        );
    }

    @logAction
    @requiredParams(['id'])
    @validateParams([
        {
            name: 'id',
            type: EValidateType.CHAR_AND_DIGIT,
        },
        {
            name: 'blankId',
            type: EValidateType.CHAR_AND_DIGIT,
        },
    ])
    downloadBlankFromTrainApi(req: Request, res: Response): void {
        const {id, blankId} = req.query;

        this.trainsService
            .downloadBlankFromTrainApi({orderId: id, blankId})
            .then(
                handlePdfFileStream(res, {
                    fileName: [id, blankId].filter(s => s).join('-'),
                }),
                handleError(req, res),
            );
    }

    @logAction
    @requestMethod('GET')
    testContextToken(req: Request, res: Response): void {
        this.trainsService
            .testContextToken(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('GET')
    ticket(req: Request, res: Response): void {
        const {voucherFileId: id} = req.params;
        const {blankId} = req.query;

        this.trainsService
            .downloadBlank({
                id,
                blankId,
            })
            .then(
                handlePdfFileStream(res, {
                    fileName: [id, blankId].filter(s => s).join('-'),
                }),
            )
            .catch(() => {
                res.redirect(
                    internalUrl(URLs.trainsAuthorizeTicket, {
                        orderId: id,
                        retpath: req.url,
                    }),
                );
            });
    }

    @logAction
    @requestMethod('GET')
    @validateParams([
        {
            name: 'slug',
            type: EValidateType.ENUM,
            data: EFirmTrain,
        },
    ])
    getFirmTrain(req: Request, res: Response): void {
        this.trainsService
            .getFirmTrain(req.query)
            .then(handleResponse(res), handleError(req, res));
    }

    @logAction
    @requestMethod('GET')
    @validateParams([
        {
            name: 'fromSlug',
            type: EValidateType.CHAR_AND_DIGIT,
        },
        {
            name: 'toSlug',
            type: EValidateType.CHAR_AND_DIGIT,
        },
    ])
    getDirectionSearch(req: Request, res: Response): void {
        this.trainsService
            .getDirectionSearch(req.query)
            .then(handleResponse(res), handleError(req, res));
    }
}

function handleError(req: Request, res: Response): (ex: Error) => void {
    return (ex: Error): void => {
        let status = HTTP_INTERNAL_ERROR_CODE;

        if (isHttpClientError(ex)) {
            status = ex?.response?.status ?? status;
        } else if (isBadRequestError(ex)) {
            status = HTTP_BAD_REQUEST_CODE;
        }

        req.utils.logError('TrainsApiController error', ex, {status});

        res.status(status).send({
            ...getOrderError(ex),
            originMessage: ex.message,
        });
    };
}

export default makeProxyInvoker(TrainsApiController);
