import {EOrderErrorType} from 'projects/trains/constants/orderErrors';

import {IOrderError} from 'reducers/trains/order/types';

import * as i18nBlock from 'i18n/trains-error-page-order';

export interface IRawPartnerError {
    type: EOrderErrorType;
    message?: string;
    description?: string;
    data?: {
        code?: number;
        message?: string;
        description?: string;
        // eslint-disable-next-line camelcase
        message_params?: string[];
    };
}

/**
 * Возвращает текст ошибки
 */
export function getMessage(error: IRawPartnerError): string | undefined {
    const {message, type, data} = error;

    if (message) {
        return message;
    }

    if (data) {
        return data.message || data.description;
    }

    if (type === EOrderErrorType.COMMON) {
        return i18nBlock.commonDashErrorDashDescription();
    }

    return undefined;
}

/**
 * Возвращает дополнительные параметры ошибки
 */
export function getParams(error: IRawPartnerError): string[] | undefined {
    // eslint-disable-next-line camelcase
    return error?.data?.message_params;
}

/**
 * Взвращает объект ошибки с разобранными полями
 * Необходимо в целях унификации доступа к данным ошибок -
 * у УФС и ИМ описание ошибок приходит в разных полях (description - УФС, message + message_params - ИМ)
 */
export default function fillError(error: IRawPartnerError): IOrderError {
    const params = getParams(error);
    const message = getMessage(error);

    return {
        ...error,
        type: error.type,
        params,
        message,
        description: error.description,
    };
}
