import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {isTripBlockType} from 'types/trips/ETripBlockType';

import {logAction} from 'server/utilities/decorators/logAction';
import requestMethod from 'server/utilities/decorators/requestMethod';
import {
    EValidateType,
    validateParams,
} from 'server/utilities/decorators/validateParams';

import {IDependencies} from 'server/getContainerConfig';
import {TripsService} from 'server/services/TripsService/TripsService';
import {handleResponse} from 'server/controllers/handleResponse';
import {handleError} from 'server/controllers/handleError';

export class TripsApiController {
    private tripsService: TripsService;

    constructor({tripsService}: IDependencies) {
        this.tripsService = tripsService;
    }

    @logAction
    @requestMethod('GET')
    getActiveTrips(req: Request, res: Response): void {
        this.tripsService
            .getActiveTrips()
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    @validateParams([
        {
            name: 'pastLimit',
            type: EValidateType.DIGIT,
        },
    ])
    getTrips(req: Request, res: Response): void {
        const {query} = req;

        this.tripsService
            .getTrips({pastLimit: query.pastLimit})
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @requestMethod('GET')
    @validateParams([
        {
            name: 'continuationToken',
            type: EValidateType.CHAR_AND_DIGIT,
        },
        {
            name: 'limit',
            type: EValidateType.DIGIT,
        },
    ])
    getMoreTrips(req: Request, res: Response): void {
        const {query} = req;

        this.tripsService
            .getMoreTrips({
                continuationToken: query.continuationToken,
                limit: query.limit,
            })
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @validateParams([
        {
            name: 'tripId',
            type: EValidateType.CHAR_AND_DIGIT,
        },
    ])
    getTrip(req: Request, res: Response): void {
        const {query} = req;

        this.tripsService
            .getTrip({tripId: query.tripId})
            .then(handleResponse(res), handleError(res));
    }

    @logAction
    @validateParams([
        {
            name: 'tripId',
            type: EValidateType.CHAR_AND_DIGIT,
        },
    ])
    getTripAsyncBlock(req: Request, res: Response): void {
        const {query} = req;

        const {tripId} = query;

        const blockTypes = Array.isArray(query.blockTypes)
            ? query.blockTypes.filter(isTripBlockType)
            : [];

        this.tripsService
            .getTripAsyncBlock({
                tripId,
                blockTypes,
            })
            .then(handleResponse(res), handleError(res));
    }
}
