import _getFp from 'lodash/fp/get';
import {ParsedQuery} from 'query-string';
import {asClass, asValue, Resolver} from 'awilix';
import {Span} from 'opentracing';

import {
    ECrowdTestHeaders,
    ETestHeaders,
    EXredirectHeaders,
} from 'server/constants/headers';
import {MOCK_COOKIE_NAME as MOCK_TRIPS_COOKIE_NAME} from 'server/api/TripsApi/constants/mockCookieName';
import {TPlatform} from 'constants/platforms/TPlatforms';

import {EWhiteLabelPartner} from 'types/common/whiteLabel/EWhiteLabelPartner';
import {
    BlackboxInfo,
    CoreConfig,
    CoreTelemetrySendStats,
    CoreUtils,
    ICookies,
    Request,
    Response,
} from '@yandex-data-ui/core/lib/types';
import {IUserSplit, TRawExperiments} from './providers/experiments/types';
import {IUATraits} from 'types/common/IUATraits';
import {TBlackboxUserInfo} from 'types/common/userInfo/TBlackboxUserInfo';
import IAttributionData from 'server/utilities/DataStorage/AttributionData/types/IAttributionData';
import IUserData from 'server/utilities/DataStorage/UserData/types/IUserData';
import {TSrcParams} from 'types/TSourceParams';
import {EApiEntry} from 'types/EApiEntry';
import {IGeobaseProvider} from 'server/providers/geoBase/types';
import {TSetCookie} from 'types/TSetCookie';

import {
    AppLogger,
    ILogger,
    TErrorLogger,
    TLogger,
} from 'server/utilities/Logger';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {IHttpClient} from 'server/utilities/HttpClient/IHttpClient';
import {HttpClient} from 'server/utilities/HttpClient/HttpClient';
import {AviaGatewayBackendApiClient} from 'server/utilities/AviaBackendApiClient/AviaGatewayBackendApiClient';
import {IAviaBackendApiClient} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';
import {AviaBackendRestApiClient} from 'server/utilities/AviaBackendApiClient/AviaBackendRestApiClient';
import {
    getFileLoggerWrapperGetter,
    TFileLoggerWrapperGetter,
} from './loggers/utils/getFileLoggerWrapper';
import {getUserSplitStandalone} from './providers/experiments/utils/getUserSplitStandalone';
import {AviaBackendApiClient} from 'server/utilities/AviaBackendApiClient/AviaBackendApiClient';
import AttributionData from 'server/utilities/DataStorage/AttributionData/AttributionData';
import UserData from 'server/utilities/DataStorage/UserData/UserData';
import {parseSrcParams} from 'server/utilities/srcParams/parseSrcParams';
import {HotelsMetaParamsBuilder} from 'server/utilities/hotels/HotelsMetaParamsBuilder/HotelsMetaParamsBuilder';
import AffiliateData from 'server/utilities/DataStorage/AffiliateData/AffiliateData';
import TestRequestManager from 'server/utilities/TestRequestManager/TestRequestManager';
import {getRequestFullUrl} from 'server/utilities/request/getRequestFullUrl';
import {isE2eRequest} from 'server/utilities/request/isE2eRequest';

import {
    AviaBookingOrdersApi,
    AviaBookingVariantsApi,
} from 'server/api/AviaBookingApi/AviaBookingApi';
import {TravelersApi} from 'server/api/TravelersApi/TravelersApi';
import {TrainsApi} from 'server/api/TrainsApi/TrainsApi';
import {TrainsBookingApi} from 'server/api/TrainsBookingApi/TrainsBookingApi';
import {TrainsOfferStorageApi} from 'server/api/TrainsOfferStorageApi/TrainsOfferStorageApi';
import {TrainsSearchApi} from 'server/api/TrainsSearchApi/TrainsSearchApi';
import {RaspApi} from 'server/api/RaspApi/RaspApi';
import {RaspCacheApi} from 'server/api/RaspCacheApi/RaspCacheApi';
import {RaspSuggestsApi} from 'server/api/RaspSuggestsApi/RaspSuggestsApi';
import {GeobaseApi} from 'server/api/GeobaseApi/GeobaseApi';
import {OrdersAPI} from 'server/api/OrdersAPI/OrdersAPI';
import {HotelsSearchAPI} from 'server/api/HotelsSearchAPI/HotelsSearchAPI';
import {HotelsSearchService} from 'server/services/HotelsServices/HotelsSearchService';
import {HotelSearchAPI} from 'server/api/HotelSearchAPI/HotelSearchAPI';
import {HotelSearchService} from 'server/services/HotelsServices/HotelSearchService';
import {HotelsBookService} from 'server/services/HotelsServices/HotelsBookService';
import {HotelsBookAPI} from 'server/api/HotelsBookAPI/HotelsBookAPI';
import {HotelsFavoritesAPI} from 'server/api/HotelsFavoritesAPI/HotelsFavoritesAPI';
import {HotelsFavoritesService} from 'server/services/HotelsServices/HotelsFavoritesService';
import {PersonalizationApi} from 'server/api/PersonalizationApi/PersonalizationApi';
import {AviaBookingService} from 'server/services/AviaBookingService/AviaBookingService';
import {AccountsService} from 'server/services/AccountsService/AccountsService';
import {RaspService} from 'server/services/RaspService/RaspService';
import {TrainsService} from 'server/services/TrainsService/TrainsService';
import {GeobaseService} from 'server/services/GeobaseService/GeobaseService';
import {TicketsTravelersNotebookService} from 'server/services/TicketsTravelersNotebookService/TicketsTravelersNotebookService';
import {OrdersService} from 'server/services/OrdersService/OrdersService';
import {AviaOrderService} from 'server/services/AviaOrderService/AviaOrderService';
import {AviaTicketDaemonApi} from 'server/api/AviaTicketDaemonApi/AviaTicketDaemonApi';
import {AviaTravelersNotebookService} from 'server/services/AviaTravelersNotebookService/AviaTravelersNotebookService';
import {AviaSearchService} from 'server/services/AviaSearchService/AviaSearchService';
import {AviaPassengerExperienceApi} from 'server/api/AviaPassengerExperienceApi/AviaPassengerExperienceApi';
import {AviaPassengerExperienceService} from 'server/services/AviaPassengerExperienceService/AviaPassengerExperienceService';
import {AviaPartnersApi} from 'server/api/AviaPartnersApi/AviaPartnersApi';
import {AviaFlightStorageApi} from 'server/api/AviaFlightStorageApi';
import {AviaSuggestsApi} from 'server/api/AviaSuggestsApi/AviaSuggestsApi';
import {AviaNearService} from 'server/services/AviaNearService/AviaNearService';
import {AviaApi} from 'server/api/AviaApi';
import {AviaRouteApi} from 'server/api/AviaRouteApi/AviaRouteApi';
import {AviaSubscriptionService} from 'server/services/AviaSubscriptionService';
import {AviaFlightStorageService} from 'server/services/AviaFlightStorageService';
import {AviaGeoService} from 'server/services/AviaGeoService/AviaGeoService';
import {AviaSearchController} from 'server/controllers/aviaApiContoller/AviaSearchController';
import {TrainsSearchService} from 'server/services/TrainsSearchService/TrainsSearchService';
import {CurrenciesService} from 'server/services/CurrenciesService/CurrenciesService';
import {AviaPriceIndexApi} from 'server/api/AviaPriceIndexApi/AviaPriceIndexApi';
import {AviaPriceIndexService} from 'server/services/AviaPriceIndexService/AviaPriceIndexService';
import {TinyUrlApi} from 'server/api/TinyUrl/TinyUrlApi';
import {TinyUrlService} from 'server/services/TinyUrlService/TinyUrlService';
import {AviaRedirectService} from 'server/services/AviaRedirectService/AviaRedirectService';
import {AviaRedirectController} from 'server/controllers/aviaApiContoller/AviaRedirectController';
import {AviaFlexibleService} from 'server/services/AviaFlexibleService/AviaFlexibleService';
import {AviaVariantsService} from 'server/services/AviaVariantsService/AviaVariantsService';
import {AviaRouteService} from 'server/services/AviaRouteService/AviaRouteService';
import {BusesApi} from 'server/api/BusesApi/BusesApi';
import {BusesGeoApi} from 'server/api/BusesGeoApi/BusesGeoApi';
import {BusesTravelApi} from 'server/api/BusesTravelApi/BusesTravelApi';
import {BusesService} from 'server/services/BusesService/BusesService';
import {BunkerApi} from 'server/api/BunkerApi/BunkerApi';
import {BunkerService} from 'server/services/BunkerService/BunkerService';
import {ImageGeneratorApi} from 'server/api/ImageGeneratorApi/ImageGeneratorApi';
import {ImageGeneratorService} from 'server/services/ImageGeneratorService/ImageGeneratorService';
import {AviaBackendApi} from 'server/api/AviaBackendApi/AviaBackendApi';
import {AviaSettlementSlugApi} from 'server/api/AviaSettlementSlugApi/AviaSettlementSlugApi';
import {AviaSettlementSlugService} from 'server/services/AviaSettlementSlugService/AviaSettlementSlugService';
import {AviaGatewayApi} from 'server/api/AviaGatewayApi/AviaGatewayApi';
import {AviaService} from 'server/services/AviaService/AviaService';
import {GenericOrderApi} from 'server/api/GenericOrderApi/GenericOrderApi';
import {GenericOrderService} from 'server/services/GenericOrderService/GenericOrderService';
import {CommonApi} from 'server/api/CommonApi/CommonApi';
import {CommonService} from 'server/services/CommonService/CommonService';
import {NotifierApi} from 'server/api/NotifierApi/NotifierApi';
import {NotifierService} from 'server/services/NotifierService/NotifierService';
import {TestContextService} from 'server/services/TestContextService/TestContextService';
import {ExperimentsService} from 'server/services/ExperimentsService/ExperimentsService';
import {HotelsCrossSaleApi} from 'server/api/HotelsCrossSaleApi/HotelsCrossSaleApi';
import {HotelsCrossSaleService} from 'server/services/HotelsServices/HotelsCrossSaleService';
import {PersonalizationService} from 'server/services/PersonalizationService/PersonalizationService';
import {AviaFlightsToApi} from 'server/api/AviaFlightsToApi/AviaFlightsToApi';
import {AviaFlightsToService} from 'server/services/AviaFlightsToService/AviaFlightsToService';
import {TripsApi} from 'server/api/TripsApi/TripsApi';
import {TripsService} from 'server/services/TripsService/TripsService';
import LocalGeobase from 'server/providers/geoBase/LocalGeobase';
import HTTPGeobase from 'server/providers/geoBase/HttpGeobase';
import GeobaseProvider from 'server/providers/geoBase/GeobaseProvider';
import {TemplatorApi} from 'server/api/TemplatorApi/TemplatorApi';
import {TemplatorService} from 'server/services/TemplatorService/TemplatorService';
import {TransformSearchApi} from 'server/api/TransformSearchApi/TransformSearchApi';
import {CrossSearchService} from 'server/services/CrossSearchService/CrossSearchService';
import {AviaCountryRestrictionsApi} from 'server/api/AviaCountryRestrictionsApi/AviaCountryRestrictionsApi';
import {ReviewsApi} from 'server/api/ReviewsApi/ReviewsApi';
import {ReviewsService} from 'server/services/ReviewsService/ReviewsService';
import {AviaBookingTestContextApi} from 'server/api/AviaBookingTestContextApi/AviaBookingTestContextApi';
import {ToursApi} from 'server/api/ToursApi/ToursApi';
import {ToursService} from 'server/services/ToursService/ToursService';
import {LevelTravelApi} from 'server/api/LevelTravelApi/LevelTravelApi';
import {LevelTravelService} from 'server/services/LevelTravelService/LevelTravelService';
import {NotebookApi} from 'server/api/NotebookApi/NotebookApi';
import {NotebookService} from 'server/services/NotebookService/NotebookService';
import {ApiRequestChannel} from 'server/sockets/apiRequestsChannel';
import {WhiteLabelService} from 'server/services/WhiteLabelService/WhiteLabelService';
import {WeatherApi} from 'server/api/WeatherApi/WeatherApi';
import {WeatherService} from 'server/services/WeatherService/WeatherService';

import {HotelsExtranetApi} from './api/HotelsExtranetApi/HotelsExtranetApi';
import {HotelsExtranetService} from './services/HotelsExtranetService/HotelsExtranetService';

function createLogger(
    log: TLogger,
    logWarn: TErrorLogger,
    logError: TErrorLogger,
    extra?: any,
): ILogger {
    return new AppLogger({log, logWarn, logError, extra});
}

function createApiHostResolver(hostsDict: Record<EApiEntry, string>) {
    return (apiName: EApiEntry): string => hostsDict[apiName];
}

function createSrcParamsResolver(srcParams: TSrcParams) {
    return (apiName: EApiEntry): Record<string, string> | undefined =>
        srcParams[apiName];
}

interface ITVMTicket {
    error?: string;
    ticket?: string;
    // eslint-disable-next-line camelcase
    tvm_id: number;
}

export interface INationalSettings {
    nationalCurrency: CurrencyType;
    allowedCurrencies: CurrencyType[];
}

const useLocalGeobase = Boolean(process.env.USE_LOCAL_GEOBASE);

function createServiceTicketResolver(tickets: Record<string, ITVMTicket>) {
    return (serviceName: string): string => {
        let name = serviceName;

        // TODO: Нужно для плавного перехода к новой переменной в tvm. hotelsAPI нужно превратить в travelAPI, потому что этот api расширяется до общепортального.
        if (name === 'travelAPI') {
            name = 'hotelsAPI';
        }

        const serviceTVM = tickets[name];

        if (!serviceTVM) {
            throw new Error('No ticket for service: ' + name);
        }

        if (!serviceTVM.ticket) {
            throw new Error(
                serviceTVM.error || 'No ticket for service: ' + name,
            );
        }

        return serviceTVM.ticket;
    };
}

function createSendClickHouseStats(
    stats: CoreTelemetrySendStats,
    table: string,
): ((data: {[name: string]: string | number}) => void) | undefined {
    if (!table) {
        return undefined;
    }

    return (data: {[name: string]: string | number}): void => {
        return stats(table, data);
    };
}

const createAppConfigResolver = (utils: CoreUtils): CoreConfig =>
    _getFp<CoreUtils, 'config'>(['config'])(utils);

export default function getContainerConfig(
    req: Request,
    res: Response,
    {isSSR}: {isSSR: boolean},
): TResolverCollection<IDependencies> {
    const {
        utils,
        blackbox,
        cookies,
        query,
        userInfo,
        session,
        tvm,
        id,
        tld,
        hostname,
        userSplit,
        uatraits,
        ip,
        headers,
        rootSpan,
        isTravelApp,
        partner,
        platform,
    } = req;

    if (!tvm || !tvm.tickets) {
        throw new Error('TVM is not defined');
    }

    if (!session.yaTravelSessionUid) {
        throw new Error('Session UID is not defined');
    }

    const {log, logWarn, logError, config, stats} = utils;

    const yandexuid = cookies.yandexuid;

    const nationalSettings =
        config.nationalSettings[tld] || config.nationalSettings.default;

    const apiHostResolver = createApiHostResolver(config.servicesAPI);
    const serviceTicketResolver = createServiceTicketResolver(tvm.tickets);
    const appConfigResolver = createAppConfigResolver(utils);
    const srcParamsResolver = createSrcParamsResolver(
        userInfo?.isYandexNetwork ? parseSrcParams(query) : {},
    );

    const logger = createLogger(log, logWarn, logError, {
        yandexuid,
        login: userInfo && 'login' in userInfo && userInfo.login,
    });

    const fileLoggerWrapperGetter = getFileLoggerWrapperGetter(
        log,
        logWarn,
        logError,
        utils.config,
    );

    const sendClickHouseStats = createSendClickHouseStats(
        stats,
        config.telemetryClickHouseEnvTable,
    );

    const setCookie = res.cookie.bind(res);

    const getUserSplit = (): Promise<IUserSplit> => getUserSplitStandalone(req);

    const referrer = req.get('referrer');

    const testBuckets = userSplit && userSplit.boxes;

    const attributionData = new AttributionData({
        logger,
        query,
        cookies,
        setCookie,
        referrer,
        testBuckets,
    });

    attributionData.updateCookiesAndFillData();

    const crowdTestHostHeader = headers[ECrowdTestHeaders.X_YA_CROWD_TEST_HOST];
    const pageTokenHeader = headers[ETestHeaders.X_YA_PAGE_TOKEN];

    const geobaseProvider = new GeobaseProvider(
        useLocalGeobase ? new LocalGeobase() : new HTTPGeobase(),
        logger,
    );

    const requestUrl = getRequestFullUrl(req);
    const sessionKey = session.yaTravelSessionUid;
    const pageToken = isSSR
        ? res.locals?.csrfToken ?? null
        : pageTokenHeader && String(pageTokenHeader);

    const testRequestManager = new TestRequestManager({
        cookies,
        isSSR,
        apiRequestsChannel: req.apiRequestsChannel,
        requestUrl,
        requestPageUrl: isSSR ? requestUrl : headers.referer ?? '',
        sessionKey,
        appConfig: appConfigResolver,
        pageToken,
    });

    return {
        rootSpan: asValue(rootSpan),
        logger: asValue(logger),
        blackbox: asValue(blackbox),
        uatraits: asValue(uatraits),
        userInfo: asValue(userInfo),
        userAgent: asValue(headers['user-agent']),
        isSSR: asValue(isSSR),
        isBot: asValue(req.isBot),
        ip: asValue(headers['x-real-ip'] as string),
        forwardedFor: asValue(headers['x-forwarded-for'] as string),
        isCrowdTestProxy: asValue(
            Boolean(headers[ECrowdTestHeaders.X_YA_CROWD_TEST_PROXY]),
        ),
        crowdTestHost: asValue(
            typeof crowdTestHostHeader === 'string'
                ? crowdTestHostHeader
                : undefined,
        ),
        isFromXredirect: asValue(
            headers[EXredirectHeaders.X_YA_TRAVEL_IS_FROM_XREDIRECT] === '1',
        ),
        isTravelApp: asValue(isTravelApp),
        platform: asValue(platform),
        query: asValue(query),
        cookies: asValue(cookies),
        referrer: asValue(referrer),
        setCookie: asValue(setCookie),
        sendClickHouseStats: asValue(sendClickHouseStats),
        yandexuid: asValue<string>(cookies.yandexuid),
        yandexGid: asValue<string>(cookies.yandex_gid),
        nationalVersion: asValue<string>(req.nationalVersion),
        nationalSettings: asValue(nationalSettings),
        preferredCurrency: asValue(
            cookies['preferred_currency'] as CurrencyType | undefined,
        ),
        isE2E: asValue(isE2eRequest(cookies)),
        mockTrips: asValue(cookies[MOCK_TRIPS_COOKIE_NAME] === 'true'),
        testRequestManager: asValue(testRequestManager),
        apiRequestsChannel: asValue(req.apiRequestsChannel),

        sessionKey: asValue(sessionKey),
        appConfig: asValue<CoreConfig>(appConfigResolver),
        requestHostname: asValue<string>(hostname),
        requestId: asValue(id),
        requestIp: asValue<string>(ip),
        requestUrl: asValue(requestUrl),
        rawExperiments: asValue(userSplit?.rawExperiments),
        testBuckets: asValue(testBuckets),
        getApiHost: asValue(apiHostResolver),
        getSrcParams: asValue(srcParamsResolver),
        getServiceTicket: asValue(serviceTicketResolver),
        attributionData: asValue(attributionData),
        whiteLabelPartnerId: asValue(partner),
        affiliateData: asClass(AffiliateData),

        userData: asClass(UserData).scoped(),
        fileLoggerWrapperGetter: asValue(fileLoggerWrapperGetter),

        httpClient: asClass(HttpClient),
        aviaGatewayBackendApiClient: asClass(AviaGatewayBackendApiClient),
        aviaBackendApiClient: asClass(AviaBackendApiClient),
        aviaBackendRestApiClient: asClass(AviaBackendRestApiClient),

        aviaGatewayApi: asClass(AviaGatewayApi),
        aviaBookingVariantsApi: asClass(AviaBookingVariantsApi),
        aviaBookingOrdersApi: asClass(AviaBookingOrdersApi),
        aviaBookingTestContextApi: asClass(AviaBookingTestContextApi),
        travelersApi: asClass(TravelersApi),
        aviaPassengerExperienceApi: asClass(AviaPassengerExperienceApi),
        aviaTicketDaemonApi: asClass(AviaTicketDaemonApi),
        aviaPartnersApi: asClass(AviaPartnersApi),
        aviaSuggestsApi: asClass(AviaSuggestsApi),
        aviaApi: asClass(AviaApi),
        aviaRouteApi: asClass(AviaRouteApi),
        aviaFlightToApi: asClass(AviaFlightsToApi),
        aviaSettlementSlugApi: asClass(AviaSettlementSlugApi),
        aviaFlightStorageApi: asClass(AviaFlightStorageApi),
        aviaSearchController: asClass(AviaSearchController),
        aviaBackendApi: asClass(AviaBackendApi),
        aviaCountryRestrictionsApi: asClass(AviaCountryRestrictionsApi),

        hotelsSearchAPI: asClass(HotelsSearchAPI),
        hotelSearchAPI: asClass(HotelSearchAPI),
        hotelSearchService: asClass(HotelSearchService),
        hotelsSearchService: asClass(HotelsSearchService),
        hotelsBookAPI: asClass(HotelsBookAPI),
        hotelsBookService: asClass(HotelsBookService),
        hotelsFavoritesAPI: asClass(HotelsFavoritesAPI),
        hotelsFavoritesService: asClass(HotelsFavoritesService),
        hotelsCrossSaleAPI: asClass(HotelsCrossSaleApi),
        hotelsCrossSaleService: asClass(HotelsCrossSaleService),
        personalizationApi: asClass(PersonalizationApi),
        personalizationService: asClass(PersonalizationService),

        hotelsMetaParamsBuilder: asClass(HotelsMetaParamsBuilder),

        hotelsExtranetApi: asClass(HotelsExtranetApi),
        hotelsExtranetService: asClass(HotelsExtranetService),

        ordersAPI: asClass(OrdersAPI),
        ordersService: asClass(OrdersService),

        genericOrderApi: asClass(GenericOrderApi),
        genericOrderService: asClass(GenericOrderService),

        aviaBookingService: asClass(AviaBookingService),
        accountsService: asClass(AccountsService),
        trainsApi: asClass(TrainsApi),
        trainsBookingApi: asClass(TrainsBookingApi),
        trainsOfferStorageApi: asClass(TrainsOfferStorageApi),
        trainsService: asClass(TrainsService),
        trainsSearchApi: asClass(TrainsSearchApi),
        trainsSearchService: asClass(TrainsSearchService),
        raspApi: asClass(RaspApi),
        raspCacheApi: asClass(RaspCacheApi),
        raspSuggestsApi: asClass(RaspSuggestsApi),
        raspService: asClass(RaspService),
        ticketsTravelersNotebookService: asClass(
            TicketsTravelersNotebookService,
        ),
        aviaService: asClass(AviaService),
        aviaTravalersNotebookService: asClass(AviaTravelersNotebookService),
        aviaPassengerExperienceService: asClass(AviaPassengerExperienceService),
        aviaSearchService: asClass(AviaSearchService),
        aviaNearService: asClass(AviaNearService),
        aviaSubscriptionService: asClass(AviaSubscriptionService),
        aviaOrderService: asClass(AviaOrderService),
        aviaFlightStorageService: asClass(AviaFlightStorageService),
        aviaGeoService: asClass(AviaGeoService),
        currenciesService: asClass(CurrenciesService),
        aviaPriceIndexApi: asClass(AviaPriceIndexApi),
        aviaPriceIndexService: asClass(AviaPriceIndexService),
        tinyUrlApi: asClass(TinyUrlApi),
        tinyUrlService: asClass(TinyUrlService),
        aviaFlexibleService: asClass(AviaFlexibleService),
        aviaRedirectService: asClass(AviaRedirectService),
        aviaRedirectController: asClass(AviaRedirectController),
        aviaVariantsService: asClass(AviaVariantsService),
        getUserSplit: asValue(getUserSplit),
        aviaRouteService: asClass(AviaRouteService),
        aviaFlightsToService: asClass(AviaFlightsToService),
        aviaSettlementSlugService: asClass(AviaSettlementSlugService),

        commonService: asClass(CommonService),
        commonApi: asClass(CommonApi),

        busesApi: asClass(BusesApi),
        busesGeoApi: asClass(BusesGeoApi),
        busesTravelApi: asClass(BusesTravelApi),
        busesService: asClass(BusesService),

        bunkerApi: asClass(BunkerApi),
        bunkerService: asClass(BunkerService),

        imageGeneratorApi: asClass(ImageGeneratorApi),
        imageGeneratorService: asClass(ImageGeneratorService),

        notifierApi: asClass(NotifierApi),
        notifierService: asClass(NotifierService),

        testContextService: asClass(TestContextService),
        experimentsService: asClass(ExperimentsService),

        geobaseApi: asClass(GeobaseApi),
        geobaseProvider: asValue(geobaseProvider),
        geobaseService: asClass(GeobaseService),

        tripsApi: asClass(TripsApi),
        tripsService: asClass(TripsService),

        templatorApi: asClass(TemplatorApi),
        templatorService: asClass(TemplatorService),

        crossSearchService: asClass(CrossSearchService),
        transformSearchApi: asClass(TransformSearchApi),

        reviewsApi: asClass(ReviewsApi),
        reviewsService: asClass(ReviewsService),

        toursApi: asClass(ToursApi),
        toursService: asClass(ToursService),

        levelTravelApi: asClass(LevelTravelApi),
        levelTravelService: asClass(LevelTravelService),

        notebookApi: asClass(NotebookApi),
        notebookService: asClass(NotebookService),

        whiteLabelService: asClass(WhiteLabelService),

        weatherApi: asClass(WeatherApi),
        weatherService: asClass(WeatherService),
    };
}

type TResolverCollection<Resolvers extends Record<string, any>> = {
    [P in keyof Resolvers]: Resolver<Resolvers[P]>;
};

export interface IDependencies {
    logger: ILogger;
    httpClient: IHttpClient;
    blackbox: BlackboxInfo;
    uatraits: IUATraits;
    sendClickHouseStats?: (data: {[name: string]: string | number}) => void;
    userInfo: TBlackboxUserInfo;
    userAgent?: string;
    ip: string;
    forwardedFor: string;
    query: ParsedQuery;
    cookies: ICookies;
    setCookie: TSetCookie;
    referrer?: string;
    isSSR: boolean;
    isBot: boolean;
    isCrowdTestProxy: boolean;
    crowdTestHost: string | undefined;
    /** Признак, что запрос пришел с домена типа xredirect.yandex.ru */
    isFromXredirect: boolean;
    /** Признак, открыто ли из приложения травела в вебвью */
    isTravelApp: boolean;
    platform?: TPlatform;
    yandexuid: string;
    yandexGid?: string;
    preferredCurrency: CurrencyType | undefined;
    sessionKey: string;
    appConfig: CoreConfig;
    requestHostname: string;
    requestId: string;
    requestIp: string;
    requestUrl: string;
    nationalVersion: string;
    nationalSettings: INationalSettings;
    rawExperiments?: TRawExperiments;
    testBuckets?: string;
    getApiHost: (apiName: EApiEntry) => string;
    getSrcParams: (apiName: EApiEntry) => Record<string, string> | undefined;
    getServiceTicket: (apiName: string) => string;
    attributionData: IAttributionData;
    affiliateData: AffiliateData;
    userData: IUserData;
    rootSpan?: Span;
    isE2E: boolean;
    mockTrips: boolean;
    testRequestManager: TestRequestManager;
    apiRequestsChannel: ApiRequestChannel;
    whiteLabelPartnerId: EWhiteLabelPartner | null;

    fileLoggerWrapperGetter: TFileLoggerWrapperGetter;
    aviaApi: AviaApi;
    aviaGatewayApi: AviaGatewayApi;
    aviaRouteApi: AviaRouteApi;
    aviaFlightToApi: AviaFlightsToApi;
    aviaSettlementSlugApi: AviaSettlementSlugApi;
    aviaBookingVariantsApi: AviaBookingVariantsApi;
    aviaBookingOrdersApi: AviaBookingOrdersApi;
    aviaBookingTestContextApi: AviaBookingTestContextApi;
    travelersApi: TravelersApi;
    aviaPassengerExperienceApi: AviaPassengerExperienceApi;
    aviaTicketDaemonApi: AviaTicketDaemonApi;
    aviaPartnersApi: AviaPartnersApi;
    aviaSuggestsApi: AviaSuggestsApi;
    aviaFlightStorageApi: AviaFlightStorageApi;
    aviaBackendApi: AviaBackendApi;
    aviaGatewayBackendApiClient: IAviaBackendApiClient;
    aviaBackendApiClient: IAviaBackendApiClient;
    aviaBackendRestApiClient: IAviaBackendApiClient;
    aviaCountryRestrictionsApi: AviaCountryRestrictionsApi;

    hotelsSearchAPI: HotelsSearchAPI;
    hotelsSearchService: HotelsSearchService;
    hotelSearchAPI: HotelSearchAPI;
    hotelSearchService: HotelSearchService;
    hotelsBookAPI: HotelsBookAPI;
    hotelsBookService: HotelsBookService;
    hotelsFavoritesAPI: HotelsFavoritesAPI;
    hotelsFavoritesService: HotelsFavoritesService;
    hotelsCrossSaleAPI: HotelsCrossSaleApi;
    hotelsCrossSaleService: HotelsCrossSaleService;

    hotelsMetaParamsBuilder: HotelsMetaParamsBuilder;

    hotelsExtranetApi: HotelsExtranetApi;
    hotelsExtranetService: HotelsExtranetService;

    personalizationApi: PersonalizationApi;
    personalizationService: PersonalizationService;

    ordersAPI: OrdersAPI;
    ordersService: OrdersService;

    genericOrderApi: GenericOrderApi;
    genericOrderService: GenericOrderService;

    accountsService: AccountsService;
    trainsApi: TrainsApi;
    trainsBookingApi: TrainsBookingApi;
    trainsOfferStorageApi: TrainsOfferStorageApi;
    trainsService: TrainsService;
    trainsSearchApi: TrainsSearchApi;
    trainsSearchService: TrainsSearchService;

    raspApi: RaspApi;
    raspCacheApi: RaspCacheApi;
    raspSuggestsApi: RaspSuggestsApi;
    raspService: RaspService;
    ticketsTravelersNotebookService: TicketsTravelersNotebookService;
    aviaService: AviaService;
    aviaBookingService: AviaBookingService;
    aviaTravalersNotebookService: AviaTravelersNotebookService;
    aviaPassengerExperienceService: AviaPassengerExperienceService;
    aviaSearchService: AviaSearchService;
    aviaNearService: AviaNearService;
    aviaSubscriptionService: AviaSubscriptionService;
    aviaOrderService: AviaOrderService;
    aviaFlightStorageService: AviaFlightStorageService;
    aviaGeoService: AviaGeoService;
    aviaSearchController: AviaSearchController;
    currenciesService: CurrenciesService;
    aviaPriceIndexApi: AviaPriceIndexApi;
    aviaPriceIndexService: AviaPriceIndexService;
    aviaRouteService: AviaRouteService;
    aviaFlightsToService: AviaFlightsToService;
    aviaSettlementSlugService: AviaSettlementSlugService;
    tinyUrlApi: TinyUrlApi;
    tinyUrlService: TinyUrlService;
    aviaFlexibleService: AviaFlexibleService;
    aviaRedirectService: AviaRedirectService;
    aviaRedirectController: AviaRedirectController;
    aviaVariantsService: AviaVariantsService;
    getUserSplit: () => Promise<IUserSplit>;

    busesApi: BusesApi;
    busesGeoApi: BusesGeoApi;
    busesTravelApi: BusesTravelApi;
    busesService: BusesService;

    commonApi: CommonApi;
    commonService: CommonService;

    bunkerApi: BunkerApi;
    bunkerService: BunkerService;

    imageGeneratorApi: ImageGeneratorApi;
    imageGeneratorService: ImageGeneratorService;

    notifierApi: NotifierApi;
    notifierService: NotifierService;

    testContextService: TestContextService;
    experimentsService: ExperimentsService;

    geobaseApi: GeobaseApi;
    geobaseProvider: IGeobaseProvider;
    geobaseService: GeobaseService;

    tripsApi: TripsApi;
    tripsService: TripsService;

    templatorApi: TemplatorApi;
    templatorService: TemplatorService;

    crossSearchService: CrossSearchService;
    transformSearchApi: TransformSearchApi;

    reviewsApi: ReviewsApi;
    reviewsService: ReviewsService;

    toursApi: ToursApi;
    toursService: ToursService;

    levelTravelApi: LevelTravelApi;
    levelTravelService: LevelTravelService;

    notebookApi: NotebookApi;
    notebookService: NotebookService;

    whiteLabelService: WhiteLabelService;

    weatherApi: WeatherApi;
    weatherService: WeatherService;
}
