import {EProjectName} from 'constants/common';

import {TLogRedirect} from './types/TLogRedirect';

import {
    AVIA_REDIRECT_LOG_NAME,
    AVIA_REDIRECT_DEBUG_LOG_NAME,
} from '../utils/aviaLoggerConstants';
import {TFileLoggerWrapperGetter} from '../../utils/getFileLoggerWrapper';
import {AviaJSONLog} from '../utils/AviaJSONLog';
import {decodeQidOrDefault} from 'projects/avia/lib/qid';
import {getPriceByNation} from '../utils/getPriceByNation';
import {getPriceByNationWithDispersedCents} from '../utils/getPriceByNationWithDispersedCents';
import {getWizardRedirKeyFromQuery} from 'utilities/url/wizard/getWizardRedirKeyFromQuery';
import {resolveContainerValue} from 'server/utilities/container/resolve';

export function createAviaRedirectLog(
    getfileLoggerWrapper: TFileLoggerWrapperGetter,
): TLogRedirect {
    const {logger, config} = getfileLoggerWrapper(
        EProjectName.AVIA,
        AVIA_REDIRECT_LOG_NAME,
    );
    const redirectLog = new AviaJSONLog(logger, config);

    return async (req, orderData, extra): Promise<void> => {
        const aviaPartnersApi = resolveContainerValue(
            req.container,
            'aviaPartnersApi',
        );

        const partner = await aviaPartnersApi.getByCode(orderData.partner);

        if (!partner || !partner.enabled) {
            const logger = resolveContainerValue(req.container, 'logger');

            logger.logWarn(
                AVIA_REDIRECT_DEBUG_LOG_NAME,
                Error('Partner is missing or disabled'),
                {
                    ...orderData,
                },
            );

            return;
        }

        const {qid} = orderData;
        const qidData = decodeQidOrDefault(qid);

        const priceCpc = getPriceByNationWithDispersedCents(
            partner.clickPriceNational,
            orderData.national,
        );

        const priceEcpc =
            orderData.clickPrice.price ||
            Number(
                getPriceByNation(
                    partner.eClickPriceNational,
                    orderData.national,
                ),
            ) / 100;

        const redirectData = {
            billing_client_id: partner.billingClientId,
            partnerId: partner.id,
            partnerCode: orderData.partnerCode,
            host: 'ticket',
            show_id: orderData.show_id,
            national: orderData.national,
            billing_order_id: partner.billingOrderId,
            qid,
            ...qidData,
            query_source: orderData.service,
            wizardRedirKey: getWizardRedirKeyFromQuery(req.query),
            search_show_id: req.query.search_show_id,
            ytp_referer: req.query.ytpReferer,
            ...orderData.clickPrice,
            price: orderData.clickPrice.price,
            /**
             * После исследования пришли к выводу, что поле price_cpc больше не актуально.
             * Чат: https://yndx-travel.slack.com/archives/C02948PPN01/p1644318621127729
             * Решили его убрать, но не сразу, а постепенно, чтобы была возможность
             * восстановить данные. Это поле занулим, и рядом запишем price_cpc_old.
             * Если к 2022.03.01 ничего не сломается, то можно это поле выпиливать.
             *
             * @deprecated
             */
            price_cpc: 0,
            /**
             * Временное поле.
             * @deprecated
             * @see price_cpc
             */
            price_cpc_old: Number(priceCpc),
            price_ecpc: Number(priceEcpc),
            ...extra,
        };

        redirectLog.log(AVIA_REDIRECT_LOG_NAME, req, redirectData);
    };
}
