import {get} from 'lodash';

import {EProjectName} from 'constants/common';

import {Request} from '@yandex-data-ui/core/lib/types';

import {AVIA_REDIRECT_SHOW_LOG_NAME} from '../utils/aviaLoggerConstants';
import {TFileLoggerWrapperGetter} from 'server/loggers/utils/getFileLoggerWrapper';
import {AviaTSKVLog} from '../utils/AviaTSKVLog';
import {getPriceByNationWithDispersedCents} from '../utils/getPriceByNationWithDispersedCents';
import {getAviaBillingPlatform} from '../utils/getAviaBillingPlatform';
import {decodeQidOrDefault} from 'projects/avia/lib/qid';
import {resolveContainerValue} from 'server/utilities/container/resolve';

import {IAviaOrderData} from 'server/controllers/aviaApiContoller/AviaRedirectController';

export type TAviaRedirectShowLog = (
    req: Request,
    orderData: IAviaOrderData,
) => void;

export function createAviaRedirectShowLog(
    getfileLoggerWrapper: TFileLoggerWrapperGetter,
): TAviaRedirectShowLog {
    const {logger, config} = getfileLoggerWrapper(
        EProjectName.AVIA,
        AVIA_REDIRECT_SHOW_LOG_NAME,
    );
    const redirectShowLog = new AviaTSKVLog(logger, config);

    return async (req, orderData) => {
        const aviaPartnersApi = resolveContainerValue(
            req.container,
            'aviaPartnersApi',
        );

        const {variantId} = req.query;
        const partner = await aviaPartnersApi.getByCode(orderData.partner);

        if (!partner) {
            // logger.error('Partner not found by code ', orderData.partner);
            return;
        }

        const baggage = orderData.baggage || {};
        const extra = {
            ...req.aviaVisitSourceIds,
            price: String(
                getPriceByNationWithDispersedCents(
                    partner.clickPriceNational,
                    req.nationalVersion,
                ),
            ),
            host: 'ticket',
            show_id: orderData.show_id,
            national: req.nationalVersion,
            pp: getAviaBillingPlatform(req),
            query_source: orderData.service,
            ...Object.keys(baggage).reduce((map, key) => {
                return Object.keys(baggage[key] || {}).reduce(
                    (baggageMap, baggageKey) => {
                        baggageMap[`baggage_${key}_${baggageKey}`] = get(
                            baggage,
                            [key, baggageKey],
                        );

                        return baggageMap;
                    },
                    map,
                );
            }, {} as Record<string, any>),
            variantId,
        };
        const {qid} = orderData;
        const qidData = decodeQidOrDefault(qid);

        redirectShowLog.log(AVIA_REDIRECT_SHOW_LOG_NAME, req, {
            datasource_id: partner.billingDatasourceId,
            billing_client_id: partner.billingClientId,
            billing_order_id: partner.billingOrderId,
            qid,
            ...qidData,
            ...extra,
        });
    };
}
