import {Request} from '@yandex-data-ui/core/lib/types';
import {TStatus} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {IUserSplit} from 'server/providers/experiments/types';
import {IShowLogVariant} from '../AviaShowLog/types/IShowLogVariant';

import {TFileLoggerWrapperGetter} from 'server/loggers/utils/getFileLoggerWrapper';
import {getVariantId} from './utils/getVariantId';
import {getActionId} from './utils/getActionId';
import {getUserSplitByRequest} from 'server/providers/experiments/utils/getUserSplitByRequest';
import {TVariantPartnersInfo} from 'projects/avia/lib/logging/getPartnersInfo';

import {IAviaOrderData} from 'server/controllers/aviaApiContoller/AviaRedirectController';

import {createAviaShowLog} from '../AviaShowLog/aviaShowLog';
import {createAviaActionLog} from '../AviaActionLog/aviaActionLog';

export interface IAviaVariantsLoggers {
    logServerAction(
        req: Request,
        session: string,
        action: string,
        experiments: IUserSplit,
    ): string;
    logServerShow(
        req: Request,
        session: string,
        actionId: string,
        orderData: IAviaOrderData,
        partnersInfo?: TVariantPartnersInfo,
    ): string;
    logClientAction(req: Request): Promise<void>;
    logClientShow(req: Request): void;
}

export const createAviaVariantsLog = (
    getfileLoggerWrapper: TFileLoggerWrapperGetter,
): IAviaVariantsLoggers => {
    return {
        logServerAction(
            req: Request,
            session: string,
            action: string,
            userSplit: IUserSplit,
        ): string {
            const actionLog = createAviaActionLog(getfileLoggerWrapper);

            const actionId = getActionId(action);

            const params = {} as any;

            actionLog({req, session, actionId, action, params, userSplit});

            return actionId;
        },

        logServerShow(
            req: Request,
            session: string,
            actionId: string,
            orderData: IAviaOrderData,
            partnersInfo?: TVariantPartnersInfo,
        ): string {
            const showLog = createAviaShowLog(getfileLoggerWrapper);

            const variantId = getVariantId(orderData.partner);

            showLog(
                req,
                {
                    variantId,
                    fromId: `s${orderData.flightsData.fromStationId}`,
                    toId: `s${orderData.flightsData.toStationId}`,
                    baggage: orderData.baggage,
                    status: 'unknown' as TStatus,
                    position: 0,
                    forward: orderData.flightsData.forwardRoute,
                    backward: orderData.flightsData.backwardRoute,
                    tariff: orderData.tariffMap,
                    partner: orderData.partner,
                    partnersInfo: partnersInfo,
                },
                session,
                actionId,
            );

            return variantId;
        },

        async logClientAction(req: Request): Promise<void> {
            const actionLog = createAviaActionLog(getfileLoggerWrapper);

            const {session, action, actionId, params} = req.body;

            const userSplit = await getUserSplitByRequest(req);

            actionLog({
                req,
                session,
                actionId,
                action,
                params,
                userSplit,
            });
        },

        logClientShow(req: Request): void {
            const showLog = createAviaShowLog(getfileLoggerWrapper);

            const variants: IShowLogVariant[] = req.body.variants || [];
            const {actionId, session} = req.body;

            variants.forEach(variant => {
                showLog(req, variant, session, actionId);
            });
        },
    };
};
