import moment from 'moment';
import cuid from 'cuid';

import {Request} from '@yandex-data-ui/core/lib/types';
import {IAviaLogConfig} from 'server/loggers/avia/types/IAviaLogConfig';
import {IAviaJSONLogData} from '../types/IAviaJSONLogData';

import {ILoggerConstructorParams} from 'server/utilities/Logger/AppLogger';
import {BaseJSONLog} from 'server/loggers/utils/BaseJSONLog';
import {getWizardRedirKeyFromQuery} from 'utilities/url/wizard/getWizardRedirKeyFromQuery';

import {getAviaBillingPlatform} from './getAviaBillingPlatform';
import {getAviaServiceName} from './getAviaServiceName';

export class AviaJSONLog extends BaseJSONLog {
    constructor(
        params: ILoggerConstructorParams,
        config: Nullable<IAviaLogConfig>,
    ) {
        const {path = null} = config || {};

        super(params, path);
    }

    log(message: string, req: Request, extra?: any): void {
        const data = this.getLogData(req);

        super.log(message, {...data, ...extra});
    }

    logError(message: string, e: Error, req: Request, extra?: any): void {
        const data = this.getLogData(req);

        super.logError(message, e, {...data, ...extra});
    }

    private getLogData(req: Request): IAviaJSONLogData {
        return {
            unixtime: moment.utc().unix(),
            eventtime: moment.utc().format('YYYYMMDDHHmmss'),
            yandexuid: req.cookies.yandexuid,
            passportuid:
                (req.userInfo && 'uid' in req.userInfo && req.userInfo.uid) ||
                '',
            userip: req.ip,
            national_version: req.nationalVersion,
            pp: getAviaBillingPlatform(req),
            lang: req.lang,
            wizardRedirKey: getWizardRedirKeyFromQuery(req.query),
            user_from_geo_id: req.geoId,
            service: getAviaServiceName(req.uatraits, req.isTravelApp),
            icookie: req.cookies.i,
            reqId: req.id,
            eppid: `portal-${cuid()}`,
            ...req.aviaVisitSourceIds,
        };
    }
}
