import moment from 'moment';
import cuid from 'cuid';

import {Request} from '@yandex-data-ui/core/lib/types';
import {IAviaLogConfig} from 'server/loggers/avia/types/IAviaLogConfig';
import {IAviaTSKVLogData} from '../types/IAviaTSKVLogData';

import {BaseTSKVLog} from '../../utils/BaseTSKVLog';
import {ILoggerConstructorParams} from 'server/utilities/Logger/AppLogger';

import {getAviaServiceName} from './getAviaServiceName';

export class AviaTSKVLog extends BaseTSKVLog {
    constructor(
        params: ILoggerConstructorParams,
        config: Nullable<IAviaLogConfig>,
    ) {
        const {path = null, type = ''} = config || {};

        super(params, path, type);
    }

    log<T>(message: string, req: Request, extra?: T): void {
        const data = this.getLogData(req);

        super.log(message, {...data, ...extra});
    }

    logError<T>(message: string, e: Error, req: Request, extra?: T): void {
        const data = this.getLogData(req);

        super.logError(message, e, {...data, ...extra});
    }

    private getLogData(req: Request): IAviaTSKVLogData {
        return {
            service: getAviaServiceName(req.uatraits, req.isTravelApp),
            eventtime: moment.utc().format('YYYYMMDDHHmmss'),
            yandexuid: req.cookies.yandexuid,
            passportuid: req.blackbox.uid,
            userip: req.ip,
            national_version: req.nationalVersion,
            icookie: req.cookies.i,
            eppid: `portal-${cuid()}`,
            reqId: req.id,
        };
    }
}
