import fsExtra from 'fs-extra';
import path from 'path';

import {AppLogger} from 'server/utilities/Logger';
import {ILoggerConstructorParams} from 'server/utilities/Logger/AppLogger';
import {unknownToErrorOrUndefined} from 'utilities/error';

import {writeToLog} from './writeToLog';

const FILE_LOGGER_ERROR = 'FILE_LOGGER_ERROR';

export abstract class BaseFileLogger extends AppLogger {
    protected abstract format<T>(msg: T): string;

    private destination: string | null;

    constructor(params: ILoggerConstructorParams, destination: string | null) {
        super(params);

        this.destination = destination;

        if (this.destination) {
            fsExtra.ensureDir(path.dirname(this.destination));
        }
    }

    log(message: string, extra: any): void {
        const preparedExtra = this.getExtra(extra);

        this._log(message, preparedExtra);
        this.writeToFile(preparedExtra);
    }

    logError(message: string, e?: Error, extra?: any): void {
        const preparedExtra = this.getExtra(extra);

        this._logError(message, e, preparedExtra);
        this.writeToFile(preparedExtra);
    }

    private writeToFile(preparedExtra: any): void {
        try {
            if (this.destination) {
                const message = this.format(preparedExtra);

                writeToLog({
                    message,
                    destination: this.destination,
                    onError: e =>
                        this._logError(FILE_LOGGER_ERROR, e, preparedExtra),
                });
            }
        } catch (e) {
            this._logError(
                FILE_LOGGER_ERROR,
                unknownToErrorOrUndefined(e),
                preparedExtra,
            );
        }
    }
}
