/**
 * @typedef {import('@yandex-data-ui/core/lib/types').Request} Request
 * @typedef {import('@yandex-data-ui/core/lib/types').Response} Response
 * @typedef {import('@yandex-data-ui/core/lib/types').NextFunction} NextFunction
 */

const {camelCase} = require('lodash');

const headerPrefix = 'x-avia-state-';
const cookieName = 'avia-app-state';

/**
 * @param {string} str
 * @returns {string}
 */
function extractStateKeyFromHeader(str) {
    return str.startsWith(headerPrefix)
        ? camelCase(str.slice(headerPrefix.length))
        : '';
}

/**
 * @param {string} val
 * @returns {string}
 */
function normalizeValue(val) {
    if (val === 'true') {
        return true;
    } else if (val === 'false') {
        return false;
    }

    return val;
}

/**
 * @param {Request} req
 * @param {Response} res
 * @returns {void}
 */
function setNativeAppState(req, res) {
    const headerState = Object.keys(req.headers).reduce((acc, header) => {
        const key = extractStateKeyFromHeader(header);

        const headersVal = req.headers[header];

        if (key && headersVal) {
            acc[key] = normalizeValue(
                Array.isArray(headersVal) ? headersVal[0] : headersVal,
            );
        }

        return acc;
    }, {});

    let cookieState;

    try {
        cookieState = JSON.parse(req.cookies[cookieName]);
    } catch (e) {
        // ignore
    }

    if (Object.keys(headerState).length > 0) {
        cookieState = Object.assign(cookieState || {}, headerState);
        res.cookie(cookieName, JSON.stringify(cookieState), {httpOnly: true});
    }

    req.nativeAppState = cookieState || {};
}

/**
 * @param {Request} req
 * @param {Response} res
 * @param {NextFunction} next
 * @returns {void}
 */
function nativeAppStateMiddleware(req, res, next) {
    try {
        setNativeAppState(req, res);
    } catch (e) {
        req.utils.logError('nativeAppStateMiddleware', e);
    }

    next();
}

module.exports = nativeAppStateMiddleware;
