import {
    IGeobaseProvider,
    IGeolocationSearchData,
    TGeobaseRegion,
    TGeobaseLocation,
    TGeobaseTraits,
} from 'server/providers/geoBase/types';

import {ILogger} from 'server/utilities/Logger';
import {unknownToErrorOrUndefined} from 'utilities/error';

export default class GeobaseProvider implements IGeobaseProvider {
    private instance: IGeobaseProvider;
    private logger: ILogger;

    constructor(geobase: IGeobaseProvider, logger: ILogger) {
        this.instance = geobase;
        this.logger = logger;
    }

    async pinpointGeolocation(
        options: IGeolocationSearchData,
    ): Promise<TGeobaseLocation | null> {
        try {
            return this.instance.pinpointGeolocation(options);
        } catch (err) {
            return this.handleGeoError(
                unknownToErrorOrUndefined(err),
                'pinpointGeolocation',
                options,
            );
        }
    }

    async regionById(id: number): Promise<TGeobaseRegion | undefined | null> {
        try {
            return this.instance.regionById(id);
        } catch (err) {
            return this.handleGeoError(
                unknownToErrorOrUndefined(err),
                'regionById',
                id,
            );
        }
    }

    async regionsByType(type: number): Promise<TGeobaseRegion[] | null> {
        try {
            return this.instance.regionsByType(type);
        } catch (err) {
            return this.handleGeoError(
                unknownToErrorOrUndefined(err),
                'regionsByType',
                type,
            );
        }
    }

    async findCountry(id: number): Promise<number | null> {
        try {
            return this.instance.findCountry(id);
        } catch (err) {
            return this.handleGeoError(
                unknownToErrorOrUndefined(err),
                'findCountry',
                id,
            );
        }
    }

    async traitsByIp(ip: string): Promise<TGeobaseTraits | null> {
        try {
            return this.instance.traitsByIp(ip);
        } catch (err) {
            return this.handleGeoError(
                unknownToErrorOrUndefined(err),
                'traitsByIp',
                ip,
            );
        }
    }

    protected handleGeoError(
        err: Error | undefined,
        method: string,
        ...args: any[]
    ): null {
        this.logger.logError(`geobase error in method ${method}`, err, args);

        return null;
    }
}
