const {get: _get} = require('lodash');
const {get: _getFP} = require('lodash/fp');

const PHONE_NUMBER_ATTRIBUTE_KEY = 1;
const FORMATTED_PHONE_NUMBER_KEY = 101;
const YANDEXOID_ATTRIBUTE_KEY = 'subscription.login.669';
const BETA_TEST_ATTRIBUTE_KEY = 'subscription.login.668';
const LANGUAGE_ATTRIBUTE_KEY = 34;
const PLUS_ATTRIBUTE_KEY = 1015;

/* Check user auth */
const isAuth = status => status === 'VALID';

/**
 * Get main user email
 *
 * @param {array} addressList - all user emails
 *
 * @return {string} user main email
 **/

const getUserEmail = addressList => {
    if (addressList && addressList.length) {
        return addressList.map(_getFP('address')).filter(Boolean)[0] || '';
    }

    return '';
};

/**
 * Get user phones by attributes[PHONE_NUMBER_ATTRIBUTE_KEY, FORMATTED_PHONE_NUMBER_KEY]
 *
 * @param {array} userPhones - all user phones
 *
 * @return {object} user phones by attributes
 **/

const getUserPhoneNumbers = userPhones => {
    if (userPhones && userPhones.length) {
        const getPhones = phoneKey =>
            userPhones.map(_getFP(`attributes.${phoneKey}`)).filter(Boolean);
        const [phoneNumber = ''] = getPhones(PHONE_NUMBER_ATTRIBUTE_KEY);
        const [formattedPhoneNumber = ''] = getPhones(
            FORMATTED_PHONE_NUMBER_KEY,
        );

        return {
            phoneNumber,
            formattedPhoneNumber,
        };
    }

    return {
        phoneNumber: '',
        formattedPhoneNumber: '',
    };
};

/**
 * Prepare and normalize userInfo
 *
 * @param {array} user - express-blackbox userInfo
 *
 * @return {object} prepared userInfo
 **/

const prepareUserData = user => {
    const {
        display_name: displayName,
        'address-list': addressList,
        attributes,
        dbfields,
        phones,
    } = user;

    const yandexLogin = _get(dbfields, YANDEXOID_ATTRIBUTE_KEY);
    const betaTest = _get(dbfields, BETA_TEST_ATTRIBUTE_KEY);

    return {
        isAuth: true,
        uid: user.id,
        contacts: {
            phones: getUserPhoneNumbers(phones),
            email: getUserEmail(addressList),
        },
        avatar: _get(displayName, 'avatar'),
        displayName: _get(displayName, 'name'),
        lang: _get(attributes, LANGUAGE_ATTRIBUTE_KEY),
        havePlus: Boolean(_get(attributes, PLUS_ATTRIBUTE_KEY)),
        isStaff: Boolean(yandexLogin),
        isBetaTest: Boolean(betaTest),
        login: user.login,
        yandexLogin,
    };
};

const pickUsers = (currentUserUid, preparedUsers, user) => {
    if (!isAuth(user.status.value)) {
        return preparedUsers;
    }

    const data = prepareUserData(user);

    if (user.id === currentUserUid) {
        Object.assign(preparedUsers, data);
    } else {
        preparedUsers.users.push(data);
    }

    return preparedUsers;
};

const reduceUsers = blackbox => {
    if (isAuth(blackbox.status) && blackbox.raw) {
        const {users = [], default_uid: currentUserUid} = blackbox.raw;

        return users.reduce(pickUsers.bind(null, currentUserUid), {
            users: [],
        });
    }

    return {};
};

module.exports = {
    reduceUsers,
};
