import {get, pickBy} from 'lodash';

import {
    SOLOMON_AVIA_REDIRECT_ERROR,
    SOLOMON_AVIA_REDIRECT_INIT,
    SOLOMON_AVIA_REDIRECT_INSTANT_SUCCESS,
    TRequestPlatform,
} from 'constants/solomon/avia';

import {Request, Response} from '@yandex-data-ui/core/lib/types';
import {IUrlQuery} from 'utilities/url/types';
import {ServerDataFetcherBag} from '../types';
import {EAviaRedirectErrorReason} from 'server/loggers/avia/AviaRedirectErrorLog/types/EAviaRedirectErrorReason';

import {aviaRedirectActions} from 'reducers/avia/redirect/actions';
import {setAviaContext} from 'reducers/avia/context/actions';

import {qidToSearchForm} from 'projects/avia/lib/qid';
import {getUserSplitByRequest} from 'server/providers/experiments/utils/getUserSplitByRequest';
import {aviaURLs} from 'projects/avia/lib/urls';
import {
    getAviaPaymentContext,
    getAviaTestContext,
} from 'server/utilities/avia/getAviaTestContext';
import {getWizardRedirKeyFromQuery} from 'utilities/url/wizard/getWizardRedirKeyFromQuery';
import {getWizardTariffKeyFromQuery} from 'utilities/url/wizard/getWizardTariffKeyFromQuery';
import {getWizardFlagsFromQuery} from 'utilities/url/wizard/getWizardFlagsFromQuery';
import {resolveContainerValue} from 'server/utilities/container/resolve';

import {createAviaRedirectErrorLog} from 'server/loggers/avia/AviaRedirectErrorLog/aviaRedirectErrorLog';

import {counter as sendSolomonCounter} from '../../../tools/solomon';

export async function aviaRedirectPageController({
    dispatch,
    req,
    res,
}: ServerDataFetcherBag): Promise<void> {
    sendSolomonInitCounter(getRequestPlatform(req));

    const aviaSearchService = resolveContainerValue(
        req.container,
        'aviaSearchService',
    );
    const aviaGeoService = resolveContainerValue(
        req.container,
        'aviaGeoService',
    );
    const aviaRedirectController = resolveContainerValue(
        req.container,
        'aviaRedirectController',
    );

    const logRedirectError = createAviaRedirectErrorLog(
        resolveContainerValue(req.container, 'fileLoggerWrapperGetter'),
    );

    let qid = req.query.qid || req.query.q;
    const partner = req.query.partner || req.query.p;

    const redirectUrlParams = buildRedirectParams(req);
    const userSplit = await getUserSplitByRequest(req);

    if (!partner) {
        logRedirectError(
            req,
            new Error(EAviaRedirectErrorReason.EMPTY_PARTNER),
            userSplit,
        );

        sendSolomonErrorCounter(getRequestPlatform(req));

        return redirect(res, aviaURLs.getUrlToRedirectErrorPage(req.query));
    }

    if (!qid) {
        logRedirectError(
            req,
            new Error(EAviaRedirectErrorReason.EMPTY_QID),
            userSplit,
        );

        sendSolomonErrorCounter(getRequestPlatform(req));

        return redirect(res, aviaURLs.getUrlToRedirectErrorPage(req.query));
    }

    const searchForm = qidToSearchForm(qid);

    // В авиа на этот кейс отображается 404(что нелогично), но чтобы быть последовательными,
    // надо полностью валидировать query.
    // Пока перенес как в авиа, но вместо 404 страница ошибки редиректа
    if (!searchForm.fromId) {
        logRedirectError(
            req,
            new Error(EAviaRedirectErrorReason.BAD_QID),
            userSplit,
        );

        sendSolomonErrorCounter(getRequestPlatform(req));

        return redirect(res, aviaURLs.getUrlToRedirectErrorPage(req.query));
    }

    const isNotInstantPartner = getNotInstantPartners(
        get(req, ['utils', 'config', 'avia', 'notInstantPartners']),
    ).includes(partner);

    if (getWizardRedirKeyFromQuery(req.query) && !isNotInstantPartner) {
        try {
            await aviaRedirectController.instantRedirect(req, res, userSplit);
            sendSolomonInstantSuccessCounter(getRequestPlatform(req));

            return;
        } catch (e) {
            const error = e as Error;

            error.message = `${EAviaRedirectErrorReason.INSTANT_REDIRECT_FAIL}: ${error.message}`;
            logRedirectError(req, error, userSplit);

            if (res.headersSent) {
                sendSolomonErrorCounter(getRequestPlatform(req));

                return;
            }
        }
    }

    try {
        const initResult = await aviaSearchService.initSearch(searchForm);

        if (!initResult) {
            throw new Error(EAviaRedirectErrorReason.REDIRECT_INIT_FAIL);
        }

        qid = initResult.id;
    } catch (e) {
        logRedirectError(req, e as Error, userSplit);
    }

    const [partnerRes, directionsData] = await Promise.all([
        aviaSearchService.getPartner(partner),
        aviaGeoService.getDirectionsData(searchForm),
    ]);

    if (!partnerRes) {
        logRedirectError(
            req,
            new Error(EAviaRedirectErrorReason.BAD_PARTNER),
            userSplit,
        );

        sendSolomonErrorCounter(getRequestPlatform(req));

        return redirect(res, aviaURLs.getUrlToRedirectErrorPage(req.query));
    }

    if (!directionsData) {
        logRedirectError(
            req,
            new Error(EAviaRedirectErrorReason.GEOLOOKUP_ERROR),
            userSplit,
        );

        sendSolomonErrorCounter(getRequestPlatform(req));

        return redirect(res, aviaURLs.getUrlToRedirectErrorPage(req.query));
    }

    const dataToRedux = {
        qid: qid || null,
        redirectUrl: aviaURLs.getUrlToRedirectUrlApi(redirectUrlParams),
        errorRedirectUrl: aviaURLs.getUrlToRedirectErrorPage(redirectUrlParams),
        partner: partnerRes,
    };
    const {from, to} = directionsData;

    dispatch(aviaRedirectActions.setRedirectData(dataToRedux));
    dispatch(
        setAviaContext({
            ...searchForm,
            fromName: from.settlement?.title || from.station?.title || '',
            toName: to.settlement?.title || to.station?.title || '',
            from: null,
            to: null,
        }),
    );
}

function buildRedirectParams(req: Request): IUrlQuery {
    const {query, lang} = req;
    /* eslint-disable camelcase */
    const {
        qid,
        q,
        partner,
        p,
        forward,
        backward,
        fare_group,
        variantId,
        tariff_sign,
        ytpReferer,
        fareFamiliesHash,
        dexp,
        yaclid,
        withBaggage,
        promoId,
        utm_source,
        ddl,
        search_show_id,
        aviaBrand,
    } = query;

    const redirectUrlParams: IUrlQuery = {
        qid: qid || q,
        partner: partner || p,
        lang,
        forward,
        backward,
        fare_group,
        variantId,
        tariff_sign,
        ytpReferer,
        fareFamiliesHash,
        variantTestContext: getAviaTestContext(req),
        paymentTestContext: getAviaPaymentContext(req),
        unisearchRedirKey: getWizardRedirKeyFromQuery(query),
        unisearchTariffKey: getWizardTariffKeyFromQuery(query),
        unisearch_flags: getWizardFlagsFromQuery(query),
        dexp,
        yaclid,
        withBaggage,
        promoId,
        fromSovetnik: utm_source?.includes('sovetnik'),
        ddl,
        search_show_id,
        aviaBrand,
    };
    /* eslint-enable camelcase */

    return pickBy(redirectUrlParams, value => typeof value !== 'undefined');
}

function redirect(res: Response, to: string): void {
    res.redirect(302, to);
}

function getNotInstantPartners(partnersFromEnv: string = ''): string[] {
    return partnersFromEnv.split(',');
}

function getRequestPlatform(req: Request): TRequestPlatform {
    const isMobile = req.uatraits.isMobile;

    return isMobile ? 'mobile' : 'desktop';
}

function sendSolomonErrorCounter(platform: TRequestPlatform): void {
    sendSolomonCounter(SOLOMON_AVIA_REDIRECT_ERROR, {platform});
}

function sendSolomonInitCounter(platform: TRequestPlatform): void {
    sendSolomonCounter(SOLOMON_AVIA_REDIRECT_INIT, {platform});
}

function sendSolomonInstantSuccessCounter(platform: TRequestPlatform): void {
    sendSolomonCounter(SOLOMON_AVIA_REDIRECT_INSTANT_SUCCESS, {
        platform,
    });
}
