import {pick} from 'lodash';

import {URLs} from 'constants/urls';
import {ERedirectStatusCodes} from 'constants/redirectStatusCodes';
import {AVIA_PARAMS_KEYS} from 'projects/avia/constants/queryParams';

import {ServerDataFetcherBag} from 'server/redux/types';

import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';

import {internalUrl} from 'utilities/url';
import {isValidAviaSearchParams} from 'projects/avia/lib/search/isValidAviaSearchParams';
import {processAviaSearchParams} from 'projects/avia/lib/search/processAviaSearchParams';

import {setAviaSort} from 'server/redux/avia/setAviaSort';
import {prefillAviaContext} from 'server/redux/avia/prefillSearchForm';
import {prefillAviaPointsData} from 'server/redux/avia/prefillAviaPointsData';
import {prefillCountryRestrictions} from 'server/redux/avia/prefillCountryRestrictions';

export default async function handleSearchWithParams(
    fetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {
        res,
        req: {query},
        getState,
    } = fetcherBag;
    const rawSearchParams = pick(query, AVIA_PARAMS_KEYS);
    const searchParams = processAviaSearchParams(rawSearchParams);
    const isValidParams = isValidAviaSearchParams(searchParams, false);

    if (isValidParams) {
        await Promise.all([
            prefillAviaContext(fetcherBag).then(() =>
                prefillCountryRestrictions(fetcherBag),
            ),
            prefillAviaPointsData(fetcherBag),
            setAviaSort(fetcherBag),
        ]);
    }

    const {from, to} = aviaContextSelector(getState());
    const contextIsNotFilled = !from || !to;

    if (!isValidParams || contextIsNotFilled) {
        res.redirect(
            internalUrl(URLs.aviaSearchError, query),
            ERedirectStatusCodes.TEMPORARILY,
        );
    }
}
