import {isEmpty, pick} from 'lodash';
import {batchActions} from 'redux-batched-actions';

import {USER_SETTLEMENT} from 'projects/avia/constants/constants';
import {AVIA_PARAMS_KEYS} from 'projects/avia/constants/queryParams';

import {ServerDataFetcherBag} from 'server/redux/types';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';
import {ELandingBlockType} from 'types/avia/landing/IAviaLanding';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {EAviaClassType} from 'types/avia/EAviaClassType';
import {TSettlementKey, TStationKey} from 'types/PointKey';

import {metrikaActions} from 'reducers/common/goals/actions';
import {getAviaSearchFormActions} from 'reducers/avia/utils/fillSearchForm';
import {setAviaContext} from 'reducers/avia/context/actions';
import {AVIA_SEARCH_FORM_INITIAL_STATE} from 'reducers/avia/context/reducer';
import requestSearchSuggestsThunkAction from 'reducers/avia/searchSuggests/thunk/requestSearchSuggestsThunkAction';

import {getUserGeoLocation} from 'selectors/common/userInfoSelector';
import {aviaContextSelector} from 'selectors/avia/context/aviaContextSelector';
import {getAviaFlightsTo, getAviaRoute} from 'selectors/avia/aviaSelectors';
import {getRouteContentBlock} from 'selectors/avia/landing/routeDataSelector';
import {getFlightsToContentBlock} from 'selectors/avia/landing/flightsToDataSelector';

import {decodeQueryKey} from 'projects/avia/lib/qid';
import {processAviaSearchParams} from 'projects/avia/lib/search/processAviaSearchParams';
import {IPreparedAviaSuggestItemWithIsUniqueTitle} from 'server/services/AviaService/utilities/prepareSuggestsResponse';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';
import {getSuggestByPointKey} from 'projects/avia/lib/getSuggestByPointKey';
import {resolveContainerValue} from 'server/utilities/container/resolve';

import {prefillAviaPointsData} from 'server/redux/avia/prefillAviaPointsData';
import replaceSearchParamPlaceholders from 'server/redux/avia/replaceSearchParamPlaceholders';

interface IAviaPoint {
    id?: TSettlementKey | TStationKey;
    name?: string;
    type: ESearchFormFieldName.FROM | ESearchFormFieldName.TO;
}

async function fetchPoint(
    fetcherBag: ServerDataFetcherBag,
    point: IAviaPoint,
): Promise<IPreparedAviaSuggestItemWithIsUniqueTitle | null> {
    if (point.id) {
        try {
            const aviaGeoService = resolveContainerValue(
                fetcherBag.req.container,
                'aviaGeoService',
            );

            return getSuggestByPointKey(aviaGeoService, point.id);
        } catch (e) {}
    }

    return null;
}

const getFromPoint = (params: Partial<IAviaParams>): IAviaPoint => ({
    type: ESearchFormFieldName.FROM,
    name: params.fromName,
    id:
        params.fromId &&
        (isSettlementKey(params.fromId) || isStationKey(params.fromId))
            ? params.fromId
            : undefined,
});

const getToPoint = (params: Partial<IAviaParams>): IAviaPoint => ({
    type: ESearchFormFieldName.TO,
    name: params.toName,
    id:
        params.toId &&
        (isSettlementKey(params.toId) || isStationKey(params.toId))
            ? params.toId
            : undefined,
});

export async function prefillAviaSearchForm(
    fetcherBag: ServerDataFetcherBag,
    query = fetcherBag.req.query,
): Promise<void> {
    const {dispatch, getState, req} = fetcherBag;

    try {
        const rawSearchParams = pick(query, AVIA_PARAMS_KEYS);
        let {fromId, toId} = rawSearchParams;

        if (fromId === USER_SETTLEMENT || toId === USER_SETTLEMENT) {
            const replaced = await replaceSearchParamPlaceholders(
                fetcherBag,
                fromId,
                toId,
            );

            fromId = replaced.fromId;
            toId = replaced.toId;
        }

        const searchParams = processAviaSearchParams({
            ...rawSearchParams,
            fromId,
            toId,
        });

        if (
            Object.keys(rawSearchParams).length !==
            Object.keys(searchParams).length
        ) {
            dispatch(metrikaActions.addGoal(EAviaGoal.FORM_VALIDATION_FAILED));
        }

        if (isEmpty(rawSearchParams)) {
            const {
                avia: {searchForm},
            } = getState();

            await dispatch(
                requestSearchSuggestsThunkAction({
                    fromField: searchForm[ESearchFormFieldName.FROM],
                    toField: searchForm[ESearchFormFieldName.TO],
                    needToSetByGeoPointIfPossible: true,
                    req,
                }),
            );

            return;
        }

        const from: IAviaPoint = getFromPoint(searchParams);
        const to: IAviaPoint = getToPoint(searchParams);
        const [parsedFrom, parsedTo] = await Promise.all([
            fetchPoint(fetcherBag, from),
            fetchPoint(fetcherBag, to),
        ]);

        const fromWasNotParsed = (from.id || from.name) && !parsedFrom;
        const toWasNotParsed = (to.id || to.name) && !parsedTo;

        if (fromWasNotParsed || toWasNotParsed) {
            dispatch(metrikaActions.addGoal(EAviaGoal.FORM_PREFILL_FAILED));
        }

        dispatch(
            batchActions(
                getAviaSearchFormActions({
                    ...searchParams,
                    fromName: searchParams.fromName || parsedFrom?.title || '',
                    toName: searchParams.toName || parsedTo?.title || '',
                    from: parsedFrom || null,
                    to: parsedTo || null,
                }),
            ),
        );
    } catch (e) {}
}

export async function prefillAviaSearchFormRouteData(
    fetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {getState} = fetcherBag;

    const routeData = getAviaRoute(getState()).data;

    if (routeData) {
        const searchForm = getRouteContentBlock(
            ELandingBlockType.searchFormBlock,
            routeData,
        );

        if (!searchForm) {
            return;
        }

        await prefillAviaSearchForm(
            fetcherBag,
            searchForm.data.searchFormParams,
        );
        await prefillAviaPointsData(
            fetcherBag,
            searchForm.data.searchFormParams,
        );
    }
}

export async function prefillAviaSearchFormFlightsToData(
    fetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {getState} = fetcherBag;

    const flightsToData = getAviaFlightsTo(getState()).data;

    if (flightsToData) {
        const searchForm = getFlightsToContentBlock(
            ELandingBlockType.searchFormBlock,
            flightsToData,
        );

        if (!searchForm) {
            return;
        }

        const {geoId} = getUserGeoLocation(getState());
        const fromId = geoId ? `c${geoId}` : undefined;
        const extraParams =
            fromId === searchForm.data.searchFormParams.toId ? {} : {fromId};
        const searchFormParams = {
            ...searchForm.data.searchFormParams,
            ...extraParams,
        };

        await prefillAviaSearchForm(fetcherBag, searchFormParams);
        await prefillAviaPointsData(
            fetcherBag,
            searchForm.data.searchFormParams,
        );
    }
}

export async function prefillAviaContext(
    fetcherBag: ServerDataFetcherBag,
    query = fetcherBag.req.query,
    outdateIsValid: boolean = true,
): Promise<void> {
    const {dispatch} = fetcherBag;

    try {
        const rawSearchParams = pick(query, AVIA_PARAMS_KEYS);
        const searchParams = processAviaSearchParams(
            rawSearchParams,
            outdateIsValid,
        );

        if (isEmpty(searchParams)) {
            return;
        }

        const from: IAviaPoint = getFromPoint(searchParams);
        const to: IAviaPoint = getToPoint(searchParams);
        const [parsedFrom, parsedTo] = await Promise.all([
            fetchPoint(fetcherBag, from),
            fetchPoint(fetcherBag, to),
        ]);

        dispatch(
            setAviaContext({
                ...AVIA_SEARCH_FORM_INITIAL_STATE,
                ...searchParams,
                fromName: searchParams.fromName || parsedFrom?.title || '',
                toName: searchParams.toName || parsedTo?.title || '',
                from: parsedFrom || null,
                to: parsedTo || null,
            }),
        );
    } catch (e) {}
}

export async function prefillAviaContextByQKey(
    fetcherBag: ServerDataFetcherBag,
    query = fetcherBag.req.query,
): Promise<void> {
    try {
        const decodedQueryKey = decodeQueryKey(query.qkey);

        await prefillAviaContext(fetcherBag, {
            ...decodedQueryKey,
            klass:
                decodedQueryKey.klass === EAviaClassType.BUSINESS
                    ? EAviaClassType.BUSINESS
                    : EAviaClassType.ECONOMY,
            adult_seats: `${decodedQueryKey.adult_seats}`,
            children_seats: `${decodedQueryKey.children_seats}`,
            infant_seats: `${decodedQueryKey.infant_seats}`,
        });
    } catch (e) {}
}

export async function prefillAviaContextFromRouteData(
    fetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {getState} = fetcherBag;

    const routeData = getAviaRoute(getState()).data;

    if (routeData) {
        const dynamicsBlock = getRouteContentBlock(
            ELandingBlockType.dynamicsBlock,
            routeData,
        );

        if (!dynamicsBlock) {
            return;
        }

        await prefillAviaContext(fetcherBag, dynamicsBlock.data);
    }
}

export async function prefillAviaContextFromFlightsToData(
    fetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {getState} = fetcherBag;

    const flightsToData = getAviaFlightsTo(getState()).data;

    if (flightsToData) {
        const dynamicsBlock = getFlightsToContentBlock(
            ELandingBlockType.dynamicsBlock,
            flightsToData,
        );

        if (!dynamicsBlock) {
            return;
        }

        const {geoId} = getUserGeoLocation(getState());
        const fromId = geoId ? `c${geoId}` : undefined;
        const extraParams = fromId === dynamicsBlock.data.toId ? {} : {fromId};
        const params = {
            ...dynamicsBlock.data,
            ...extraParams,
        };

        await prefillAviaContext(fetcherBag, params);
    }
}

export function fillSearchFormByContext({
    getState,
    dispatch,
}: ServerDataFetcherBag): void {
    dispatch(
        batchActions(getAviaSearchFormActions(aviaContextSelector(getState()))),
    );
}
