import {uniqBy} from 'lodash';

import {
    DEFAULT_ADULTS_COUNT,
    DEFAULT_AVIA_CLASS,
    DEFAULT_INFANTS_COUNT,
    DEFAULT_CHILDREN_COUNT,
} from 'constants/avia';

import {ServerDataFetcherBag} from 'server/redux/types';
import {IVariantPrice} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {decodeQidOrDefault, qidToSearchForm} from 'projects/avia/lib/qid';
import {aviaURLs} from 'projects/avia/lib/urls';
import {resolveContainerValue} from 'server/utilities/container/resolve';

import {prefillAviaContext} from 'server/redux/avia/prefillSearchForm';

export async function aviaRedirectErrorController(
    fetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {req, res} = fetcherBag;
    const aviaOrderService = resolveContainerValue(
        req.container,
        'aviaOrderService',
    );

    const {qid, partner, forward, backward, withBaggage} = req.query;

    try {
        const decodedQid = decodeQidOrDefault(qid);
        const data = await aviaOrderService.getOrder({
            ...decodedQid,
            adult_seats:
                `${decodedQid.adult_seats}` || `${DEFAULT_ADULTS_COUNT}`,
            children_seats:
                `${decodedQid.children_seats}` || `${DEFAULT_CHILDREN_COUNT}`,
            infant_seats:
                `${decodedQid.infant_seats}` || `${DEFAULT_INFANTS_COUNT}`,
            klass: decodedQid.klass || DEFAULT_AVIA_CLASS,
            forward,
            backward,
            qid,
        });

        if (data && data.variants.fares.length) {
            const [fare] = data.variants.fares;
            const {prices} = fare;
            const uniqPartnerPrices = uniqBy<IVariantPrice>(
                prices,
                (item: IVariantPrice) => item.partnerCode,
            );

            // Редиректим на покупку, в том случае, если на покупке будет больше 1 партнера или 1 партнер,
            // но не тот, по которому пользователь пытался перейти
            if (
                uniqPartnerPrices.length > 1 ||
                (uniqPartnerPrices.length === 1 &&
                    uniqPartnerPrices[0].partnerCode !== partner)
            ) {
                return res.redirect(
                    302,
                    aviaURLs.getOrderUrl({
                        ...decodedQid,
                        forward,
                        backward,
                        partner,
                        fromError: 1,
                        baggage: withBaggage,
                    }),
                );
            }
        }
    } catch (e) {}

    // Заполняем форму только в том случае если нам не нужно средиректить на покупку
    await prefillAviaContext(fetcherBag, qidToSearchForm(qid));
}
