import {ServerDataFetcherBag} from 'server/redux/types';

import {
    setToursSearchFormAdultsFieldAction,
    setToursSearchFormChildrenAgesFieldAction,
    setToursSearchFormFlexibleDatesAction,
    setToursSearchFormFlexibleNightsAction,
    setToursSearchFormFromFieldAction,
    setToursSearchFormNightsFieldAction,
    setToursSearchFormStartDateFieldAction,
    setToursSearchFormToFieldAction,
} from 'reducers/tours/searchForm/actions';

import {getUserGeoLocation} from 'selectors/common/userInfoSelector';

import {logError} from 'utilities/logger/logError';
import parseAutocompleteResponse from 'utilities/tours/parseAutocompleteResponse';
import {
    parseAdults,
    parseChildrenAges,
} from 'projects/hotels/utilities/validateSearchHotelsParams/validateSearchHotelsParams';
import findMatchedDeparture from 'utilities/tours/findMatchedDeparture';

import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

import {levelTravel} from 'serviceProvider/tours/levelTravelProvider/levelTravelProvider';

export async function prefillToursSearchForm(
    fetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {dispatch, getState, req} = fetcherBag;
    try {
        const state = getState();

        const {
            departure_from,
            destination,
            departure_date,
            adults,
            nights_count,
            childrenAges,
            flexNights,
            flexDates,
        }: Partial<Record<string, string>> = req.query;

        const departuresResponse = await levelTravel
            .provider(req?.container)
            .getDepartures({prioritized_count: 10});

        const geoLocation = getUserGeoLocation(state);

        let from_city = departure_from;

        if (departure_from) {
            const matchedDepartureFrom = findMatchedDeparture({
                departures: departuresResponse.departures,
                departureName: departure_from,
            });

            if (matchedDepartureFrom) {
                dispatch(
                    setToursSearchFormFromFieldAction({
                        selectedValue: matchedDepartureFrom,
                        inputValue: matchedDepartureFrom.name_ru,
                        source: ESuggestSource.SEARCH_CONTEXT,
                    }),
                );
            }
        } else {
            const matchedGeo = findMatchedDeparture({
                departures: departuresResponse.departures,
                departureName: geoLocation.geoName || '',
            });

            if (matchedGeo) {
                from_city = matchedGeo.name_en;

                dispatch(
                    setToursSearchFormFromFieldAction({
                        selectedValue: matchedGeo,
                        inputValue: matchedGeo.name_ru,
                        source: ESuggestSource.SEARCH_CONTEXT,
                    }),
                );
            }
        }

        if (destination) {
            const destinationsResponse = await levelTravel
                .provider(req?.container)
                .autocomplete({
                    query: destination,
                    from_city: from_city || '',
                });

            const selectedItem =
                parseAutocompleteResponse(destinationsResponse)[0];

            dispatch(
                setToursSearchFormToFieldAction({
                    selectedValue: selectedItem,
                    inputValue: selectedItem.name_ru,
                    source: ESuggestSource.SEARCH_CONTEXT,
                }),
            );
        }

        if (departure_date) {
            dispatch(setToursSearchFormStartDateFieldAction(departure_date));
        }

        if (adults) {
            dispatch(setToursSearchFormAdultsFieldAction(parseAdults(adults)));
        }

        if (nights_count) {
            dispatch(setToursSearchFormNightsFieldAction(nights_count));
        }

        if (flexDates) {
            dispatch(setToursSearchFormFlexibleDatesAction(Boolean(flexDates)));
        }

        if (flexNights) {
            dispatch(
                setToursSearchFormFlexibleNightsAction(Boolean(flexNights)),
            );
        }

        if (childrenAges) {
            dispatch(
                setToursSearchFormChildrenAgesFieldAction(
                    parseChildrenAges(childrenAges),
                ),
            );
        }
    } catch (err) {
        logError(
            {
                message: 'levelTravelError: failed to prefetch data on SSR',
            },
            err,
        );
    }
}
