import flatMap from 'lodash/flatMap';

import {OutdatedSearchError} from 'projects/trains/lib/search/constants';
import {ORDER_STEP} from 'projects/trains/constants/orderSteps';

import {ServerDataFetcherBag} from 'server/redux/types';
import {isFilledTrainsSearchContext} from 'reducers/trains/context/types';
import {DIRECTIONS} from 'types/common/EDirection';
import {ITrainOrderApiSegment} from 'server/api/RaspApi/types/IRaspGetTrainOrderSegmentApi/models';
import {ITrainsTariffApiSegment} from 'server/api/TrainsApi/types/ITrainsGetTariffsApi/models';

import {fillFormFromContextThunkAction} from 'reducers/trains/searchForm/thunk/fillFormFromContextThunkAction';
import trainsParseContext from 'reducers/trains/context/actions/trainsParseContext';
import {
    setDirection,
    setIndex,
    setOrderStep,
} from 'reducers/trains/order/actions/view';
import fetchTrainsOrderSegmentFromTariffs from 'reducers/trains/order/thunk/fetchTrainsOrderSegmentFromTariffs';
import {
    setOrderSegment,
    setTrainDetails,
} from 'reducers/trains/order/actions/trains';

import currentSegmentDirectionAndIndexSelector from 'selectors/trains/order/currentSegmentDirectionAndIndexSelector';

import getContextFromOrderUrlParams from 'projects/trains/lib/context/getContextFromOrderUrlParams';
import isOrderStep from 'projects/trains/lib/order/steps/isOrderStep';
import {trainsURLs} from 'projects/trains/lib/urls';
import {getTrainsOrderParamsByQuery} from 'projects/trains/lib/urls/getTrainsOrderParamsByQuery/getTrainsOrderParamsByQuery';
import {ROBOT} from 'utilities/dateUtils/formats';
import {getOrderStepDescriptionByPath} from 'projects/trains/lib/urls/getOrderStepUrl';
import {unknownToErrorOrUndefined} from 'utilities/error';

import {raspService} from 'serviceProvider';

export async function fetchOrderData({
    dispatch,
    getState,
    req,
    res,
}: ServerDataFetcherBag): Promise<void> {
    try {
        const query = getTrainsOrderParamsByQuery(req.query);
        const stepDescription = getOrderStepDescriptionByPath(req.path);

        if (stepDescription.step === ORDER_STEP.PLACES) {
            dispatch(setDirection(stepDescription.direction));
            dispatch(setIndex(stepDescription.index));
        }

        await dispatch(
            trainsParseContext(
                getContextFromOrderUrlParams(query),
                req.container,
            ),
        );

        const {
            trains: {context},
        } = getState();

        if (!isFilledTrainsSearchContext(context)) {
            return;
        }

        const promises = flatMap(DIRECTIONS, direction => {
            const routes = query[direction] ?? [];

            return routes.map(async (route, index) => {
                const {when, number, fromId, toId} = route;

                if (!when || !number) {
                    return;
                }

                let segment:
                    | ITrainOrderApiSegment
                    | ITrainsTariffApiSegment
                    | null;

                try {
                    segment = await raspService
                        .provider(req.container)
                        .getTrainOrderSegment({
                            pointFrom: fromId,
                            pointTo: toId,
                            number,
                            departure: when.format(`${ROBOT}THH.mm`),
                        });
                } catch (e) {
                    segment = await dispatch(
                        fetchTrainsOrderSegmentFromTariffs(
                            {
                                pointFrom: fromId,
                                pointTo: toId,
                                when,
                                number,
                            },
                            req,
                        ),
                    );
                }

                if (!segment) {
                    return;
                }

                dispatch(
                    setOrderSegment({
                        direction,
                        index,
                        data: segment,
                    }),
                );
            });
        });

        await Promise.all(promises);

        dispatch(fillFormFromContextThunkAction());

        const orderStepMatch = req.path.match(/order\/(\w+)/);

        if (orderStepMatch) {
            const orderStep = orderStepMatch[1];

            if (isOrderStep(orderStep)) {
                dispatch(setOrderStep(orderStep));
            }
        }
    } catch (err) {
        if (err instanceof OutdatedSearchError) {
            const {
                trains: {context},
            } = getState();

            res.redirect(
                302,
                trainsURLs.getTrainsSearchUrl({
                    context: {
                        from: context.from?.slug || '',
                        to: context.to?.slug || '',
                    },
                    query: {
                        m__outdate_order_request: 1,
                    },
                }),
            );

            return;
        }

        req.utils.logError('TRAINS', unknownToErrorOrUndefined(err), {
            uid: req.cookies.yandexuid,
            level: 'renderServer',
            data: 'actionType: fetchOrderData',
        });

        const currentSegmentDirectionAndIndex =
            currentSegmentDirectionAndIndexSelector(getState());

        // Показываем ошибку. Например, если requestOrderSegment() присылает ошибку со статусом 400
        // "errors": {
        //   "wrongRequest": [
        //     "no_such_segment"
        //   ]
        // }
        dispatch(
            setTrainDetails({
                ...currentSegmentDirectionAndIndex,
                data: {
                    trainDetails: null,
                    errors: true,
                    partnerError: null,
                },
            }),
        );
    }
}
