import omit from 'lodash/omit';

import {
    OutdatedSearchError,
    PointNotFoundSearchError,
} from 'projects/trains/lib/search/constants';
import {URLs} from 'constants/urls';

import {ServerDataFetcherBag} from 'server/redux/types';
import {
    isFilledTrainsSearchContext,
    TrainsSearchContextType,
} from 'reducers/trains/context/types';

import trainsParseContext from 'reducers/trains/context/actions/trainsParseContext';

import getContextFromOrderUrlParams from 'projects/trains/lib/context/getContextFromOrderUrlParams';
import isTrainsOutdatedOrWrongWhen from 'projects/trains/lib/date/isTrainsOutdatedOrWrongWhen';
import {trainsURLs} from 'projects/trains/lib/urls';
import {unknownToErrorOrUndefined} from 'utilities/error';

function getIsOutdatedWhen(context: TrainsSearchContextType): boolean {
    return Boolean(
        context.originalWhen &&
            isTrainsOutdatedOrWrongWhen(
                context.originalWhen,
                context.from?.timezone,
            ),
    );
}

function redirectToPartiallyFilledForm({
    res,
    getState,
}: ServerDataFetcherBag): void {
    const {
        trains: {context},
    } = getState();

    const isOutdatedWhen = getIsOutdatedWhen(context);

    res.redirect(
        302,
        trainsURLs.getIndexPageUrl({
            fromSlug: context.from?.slug || undefined,
            toSlug: context.to?.slug || undefined,
            when:
                (isOutdatedWhen ? undefined : context.originalWhen) ||
                undefined,
            m__outdate_search_request: isOutdatedWhen ? 1 : undefined,
        }),
    );
}

/*
 * Редиректим запросы с колдунщиков по типу /trains/search/?fromId=c54&toId=c213&when=2018-10-16&&utm_source=some
 * на поиск со слагами /yekaterinburg--moscow/2018-10-16?utm_source=some
 */
export default async function trainsRedirectOldSearch(
    serverDataFetcherBag: ServerDataFetcherBag,
): Promise<void> {
    const {req, res, dispatch, getState} = serverDataFetcherBag;

    try {
        await dispatch(
            trainsParseContext(
                getContextFromOrderUrlParams(req.query),
                req.container,
            ),
        );
    } catch (err) {
        if (err instanceof PointNotFoundSearchError) {
            redirectToPartiallyFilledForm(serverDataFetcherBag);

            return;
        } else if (!(err instanceof OutdatedSearchError)) {
            req.utils.logError('TRAINS', unknownToErrorOrUndefined(err), {
                uid: req.cookies.yandexuid,
                level: 'renderServer',
                data: 'actionType: redirectOldSearch',
            });

            res.redirect(302, URLs.trains);

            return;
        }
    }

    const {
        trains: {context},
    } = getState();

    if (!isFilledTrainsSearchContext(context)) {
        redirectToPartiallyFilledForm(serverDataFetcherBag);

        return;
    }

    const urlToRedirect = trainsURLs.getTrainsSearchUrl({
        context: {
            ...context,
            from: context.from.slug,
            to: context.to.slug,
        },
        query: omit(req.query, ['fromId', 'toId', 'when']),
    });

    res.redirect(301, urlToRedirect);
}
