import fs from 'fs';
import path from 'path';
import serialize from 'serialize-javascript';
import {FunctionComponent} from 'react';

import {TExperiments} from 'server/providers/experiments/types';

import {IDevice} from 'reducers/common/commonReducerTypes';

import getPlatform from 'utilities/deviceType/getPlatform';
import readAndJoinScriptsSync from 'utilities/readAndJoinScriptsSync/readAndJoinScriptsSync';

/* Prepare paths */
const rootPath = fs.realpathSync(process.cwd());
const errorCounterScripts = [
    'interfaceOverRum.min.js',
    'implementation.min.js',
    'filters.min.js',
    'logError.min.js',
    'logAjaxError.min.js',
    'traceUnhandledRejection.min.js',
].map(fileName =>
    path.join(
        rootPath,
        'node_modules',
        '@yandex-int',
        'error-counter',
        'dist',
        fileName,
    ),
);

const errorCounterScriptsContent = readAndJoinScriptsSync(errorCounterScripts);

interface IYandexErrorsProps {
    appVersion: string;
    geoId: number | undefined;
    appEnv: string | undefined;
    nonce: string;
    deviceType: IDevice;
    experiments: TExperiments;
}

interface IYandexErrorFilter {
    [key: string]: string;
}

/** https://a.yandex-team.ru/arcadia/frontend/packages/rum-counter */
interface IErrorCounterSettings {
    version: string;
    project: string;
    page: string;
    env: string | undefined;
    unhandledRejection: boolean;
    traceUnhandledRejection: boolean;
    platform: string;
    debug: boolean;
    region: number | undefined;
    filter: {
        stack?: IYandexErrorFilter;
        message?: IYandexErrorFilter;
        url?: IYandexErrorFilter;
    };
    experiments: string[];
    transform?(error: IRumErrorOptions): IRumErrorOptions;
}

const YandexErrors: FunctionComponent<IYandexErrorsProps> = props => {
    const {geoId, appEnv, nonce, deviceType, experiments, appVersion} = props;
    const enabledExperiments = Object.entries(experiments)
        .filter(([_, expValue]) => expValue)
        .map(([expKey]) => expKey);

    const errorCounterSettings: IErrorCounterSettings = {
        version: appVersion,
        project: 'travel',
        page: 'index',
        // В error-booster'е фиксированный enum для окружений и "beta" там нет
        // @see https://a.yandex-team.ru/arcadia/frontend/packages/rum-counter
        env: appEnv === 'beta' ? 'development' : appEnv,
        unhandledRejection: true,
        traceUnhandledRejection: Math.random() < 0.05,
        platform: getPlatform(deviceType.isMobile),
        experiments: enabledExperiments,
        debug: false,
        region: geoId,
        filter: {
            message: {
                // TRAVELFRONT-1812
                ADBLOCK_OLD_OPERA:
                    'Uncaught TypeError: document.querySelectorAll\\(...\\)\\[Symbol.iterator\\] is not a function',
            },
            stack: {
                ADBLOCK_OLD_OPERA: '^$',
            },
        },
    };

    return (
        <>
            <script
                nonce={nonce}
                dangerouslySetInnerHTML={{__html: errorCounterScriptsContent}}
            />
            <script
                nonce={nonce}
                dangerouslySetInnerHTML={{
                    __html: `Ya.Rum.initErrors({
                        ...${serialize(errorCounterSettings)},
                        transform: (error) => ({
                            ...error,
                            service: 'front.portal',
                        }),
                    })`,
                }}
            />
        </>
    );
};

export default YandexErrors;
