/* global __DEV__ */
/* eslint no-unused-vars: "off" */
/* eslint @typescript-eslint/no-unused-vars: "off" */
import 'core-js/stable';
import React, {Component} from 'react';
import {renderToString} from 'react-dom/server';
import {StaticRouter as Router} from 'react-router-dom';
import PropTypes from 'prop-types';
import {Provider} from 'react-redux';
import 'moment-timezone';
import {get as _get} from 'lodash';
import {v4 as uuidV4} from 'uuid';
import {ChunkExtractorManager} from '@loadable/server';

import {SECOND} from 'utilities/dateUtils/constants';

import createReduxStore from 'reducers/createServerReduxStore';
import {resetState} from 'reducers/utils/resetState/actions';
import {setApiRequestInfoItems} from 'reducers/testControlPanel/actions';

import {getChunkExtractor} from './utilities/loadable/getChunkExtractor';
import {setDateLocale} from 'utilities/dateUtils';
import locales from 'utilities/dateUtils/locales';
import webvisorIsEnabled from 'server/utilities/metrika/webvisorIsEnabled';
import appData from 'utilities/appData/appData';
import setJaegerDebugTag from './utilities/jaeger/setJaegerDebugTag';
import getCriticalCss from 'server/utilities/getCriticalCss';
import addDoctypeString from 'server/utilities/html/addDoctypeString';
import getAppVersionFromReq from 'server/utilities/getAppVersionFromReq';
import {delay} from 'utilities/async/delay';
import {resolveContainerValue} from 'server/utilities/container/resolve';
import {responseByRouterContext} from './utilities/renderServer/responseByRouterContext';

import App from 'containers/App/App';

import HtmlContainer from './render/components/HtmlContainer/HtmlContainer';

import {ServerFetchDataProvider} from 'contexts/ServerFetchDataContext';

import getContainerConfig from 'server/getContainerConfig';
import fetchUserGeoLocation from 'server/redux/common/fetchUserGeoLocation';
import fetchWhiteLabelConfig from 'server/redux/common/fetchWhiteLabelConfig';

import fetchCommonReduxInfo from './redux/common/fetchCommonInfo';
import fetchProjectReduxInfo from './redux/common/fetchProjectReduxInfo';
import fetchEnvironmentConfig from './redux/common/fetchEnvironmentConfig';
import fetchAviaReduxInfo from './redux/avia/fetchAviaInfo';

/* Constants */
const REDIRECT_STATUSES = [301, 302];
const SCRIPT_PROPS = {
    // Логирование ошибок - https://a.yandex-team.ru/arcadia/frontend/packages/rum-counter
    crossOrigin: 'anonymous',
};

const BALANCER_REQUEST_TIMEOUT = 10 * SECOND;
// Оставляем 500мс на накладные расходы запроса
const REQUEST_TIME_LIMIT = BALANCER_REQUEST_TIMEOUT - 500;

/* Helpers */
function registerContainer(req, res) {
    req.container.register(getContainerConfig(req, res, {isSSR: true}));
}

setDateLocale(locales.RU);

/* Server App */
class ServerApp extends Component {
    static propTypes = {
        req: PropTypes.shape({
            url: PropTypes.string,
        }),
        reduxStore: PropTypes.object,
        routerContext: PropTypes.object,
    };

    render() {
        const {req, reduxStore, routerContext} = this.props;

        return (
            <React.StrictMode>
                <Provider store={reduxStore}>
                    <Router location={req.url} context={routerContext}>
                        <App />
                    </Router>
                </Provider>
            </React.StrictMode>
        );
    }
}

const DEFAULT_EXTRACTOR = getChunkExtractor(['rum', 'client']);

/* Main render */
export default () => async (req, res) => {
    try {
        const reduxStore = createReduxStore();

        const fetchDataList = [];
        const serverDataFetcherBag = {
            req,
            res,
            dispatch: reduxStore.dispatch,
            getState: reduxStore.getState,
        };

        const appVersion = getAppVersionFromReq(req);
        const env = _get(req, 'utils.config.env', 'production');
        const jaegerDebugId = setJaegerDebugTag(req);
        const uid = uuidV4();
        const applicationData = {
            appEnv: env,
            jaegerDebugId,
            uid,
        };

        registerContainer(req, res);

        // Нужно для правильного рендера react приложения, т.к. некоторые пути могут зависеть от features
        fetchCommonReduxInfo({req, res}, reduxStore);
        fetchAviaReduxInfo({req, res}, reduxStore);

        await fetchUserGeoLocation(serverDataFetcherBag);
        await fetchWhiteLabelConfig(serverDataFetcherBag);

        appData.data = applicationData;

        /**
         * ChunkExtractorManager дает понимание серверу откуда зарезолвить loadable компоненты
         * при их первом рендере
         */
        const initJSX = (
            <ChunkExtractorManager extractor={DEFAULT_EXTRACTOR}>
                <ServerFetchDataProvider fetchDataList={fetchDataList}>
                    <ServerApp
                        req={req}
                        reduxStore={reduxStore}
                        routerContext={{}}
                    />
                </ServerFetchDataProvider>
            </ChunkExtractorManager>
        );

        /* Fill reduxAsyncServerActions */
        renderToString(initJSX);

        const reduxStateSnapshot = reduxStore.getState();
        const timePassedSinceRequestStart = Date.now() - req.startTime;

        let isTimedOut = false;

        await Promise.race([
            fetchProjectReduxInfo(serverDataFetcherBag, fetchDataList),
            (async () => {
                await delay(REQUEST_TIME_LIMIT - timePassedSinceRequestStart);

                isTimedOut = true;
            })(),
        ]);

        /**
         * Если не успели уложиться в таймаут балансера, ревертим все изменения в сторе
         * Страница будет загружаться как если бы пользователь перешел на нее в браузере с другой страницы
         */
        if (isTimedOut) {
            reduxStore.dispatch(resetState(reduxStateSnapshot));

            req.utils.logError('Fetch endpoints timeout', undefined, {
                requestUrl: req.originalUrl,
            });
        }

        if (env !== 'production') {
            const testRequestManager = resolveContainerValue(
                req.container,
                'testRequestManager',
            );

            reduxStore.dispatch(
                setApiRequestInfoItems(
                    testRequestManager.getApiRequestInfoItems(),
                ),
            );
        }

        if (REDIRECT_STATUSES.includes(res.statusCode)) {
            return null;
        }

        fetchEnvironmentConfig(req, reduxStore);

        const reduxState = reduxStore.getState();
        const routerContext = {};

        const extractor = getChunkExtractor(['rum', 'client']);

        const jsx = extractor.collectChunks(
            <ServerApp
                req={req}
                reduxStore={reduxStore}
                routerContext={routerContext}
            />,
        );

        // Нужно отрендерить приложение перед HtmlContainer, чтобы в Helmet'e была актуальная метаинформация.
        const appString = renderToString(jsx);

        const scriptElements = extractor.getScriptElements(SCRIPT_PROPS);
        const preloadScriptElements = extractor
            .getLinkElements(SCRIPT_PROPS)
            .filter(el => el.props.as === 'script');

        const criticalCss = getCriticalCss(extractor);

        const html = addDoctypeString(
            renderToString(
                <HtmlContainer
                    reduxState={reduxState}
                    scriptElements={scriptElements}
                    criticalCss={criticalCss}
                    preloadScriptElements={preloadScriptElements}
                    reqPath={req.path}
                    reqQuery={req.query}
                    appString={appString}
                    nonce={req.nonce}
                    appData={applicationData}
                    appVersion={appVersion}
                    webvisor={webvisorIsEnabled(
                        req.url,
                        _get(req, 'utils.config.webvisor'),
                    )}
                />,
            ),
        );

        return responseByRouterContext(res, routerContext, html);
    } catch (e) {
        req.utils.logError('An error occurred while rendering', e);

        if (__DEV__) {
            return res.status(500).send(
                addDoctypeString(`
                <html>
                    <head>
                        <title>Error</title>
                    </head>
                    <body>
                        <pre>${e.stack || e.message || e}</pre>
                    </body>
                </html>
            `),
            );
        }

        return res.status(500).send();
    }
};
