import {BlackboxInfo} from '@yandex-data-ui/core/build/types';
import {
    IPassengerWithBonusCardsDTO,
    IPassengerWithDocumentsAndBonusCardsDTO,
    IPassengerWithDocumentsDTO,
} from 'server/api/TravelersApi/types/IPassengerDTO';

import {getBlackboxDefaultEmail} from 'server/utilities/getBlackboxDefaultEmail';
import {getBlackboxDefaultPhone} from 'server/utilities/getBlackboxDefaultPhone';

import {IDependencies} from 'server/getContainerConfig';
import {TravelersApi} from 'server/api/TravelersApi/TravelersApi';

export interface IAccountService {
    getPassengersWithDocuments: () => Promise<IPassengerWithDocumentsDTO[]>;
    getPassengersWithDocumentsAndBonusCards: () => Promise<
        IPassengerWithDocumentsAndBonusCardsDTO[]
    >;
}

export class AccountsService implements IAccountService {
    private readonly uid?: string;

    private readonly travelersApi: TravelersApi;

    private readonly blackbox: BlackboxInfo;

    constructor({travelersApi, userInfo, blackbox}: IDependencies) {
        this.travelersApi = travelersApi;
        this.blackbox = blackbox;
        this.uid = userInfo && 'uid' in userInfo ? userInfo.uid : undefined;
    }

    async getPassengersWithDocuments(): Promise<IPassengerWithDocumentsDTO[]> {
        return this.getPassengersWithData(this.uid!, ['documents']);
    }

    async getPassengersWithDocumentsAndBonusCards(): Promise<
        IPassengerWithDocumentsAndBonusCardsDTO[]
    > {
        return this.getPassengersWithData(this.uid!, [
            'documents',
            'bonus-cards',
        ]);
    }

    private async getPassengersWithData(
        uid: string,
        fields: ['documents', 'bonus-cards'] | ['bonus-cards', 'documents'],
    ): Promise<IPassengerWithDocumentsAndBonusCardsDTO[]>;
    private async getPassengersWithData(
        uid: string,
        fields: ['documents'],
    ): Promise<IPassengerWithDocumentsDTO[]>;
    private async getPassengersWithData(
        uid: string,
        fields: ['bonus-cards'],
    ): Promise<IPassengerWithBonusCardsDTO[]>;
    private async getPassengersWithData(
        uid: string,
        fields:
            | ['documents']
            | ['bonus-cards']
            | ['documents', 'bonus-cards']
            | ['bonus-cards', 'documents'],
    ): Promise<
        | IPassengerWithDocumentsAndBonusCardsDTO[]
        | IPassengerWithDocumentsDTO[]
        | IPassengerWithBonusCardsDTO[]
    > {
        const traveller = await this.travelersApi.getTraveler(uid);

        /* Если путешественник не найден - API вернет строку (html с 404) */
        if (typeof traveller === 'string') {
            const email = getBlackboxDefaultEmail(this.blackbox);

            if (!email) throw new Error('User email not found');

            const phone = getBlackboxDefaultPhone(this.blackbox) ?? '';

            await this.travelersApi.createOrUpdateTraveler(uid, {
                agree: true,
                email,
                phone,
            });
        }

        return this.travelersApi.getAllPassengers(uid, fields);
    }
}
