import {
    IBookingAllVariant,
    IBookingFlight,
    IBookingFlights,
    IBookingVariantsResponse,
    IFlightDTO,
} from 'server/api/AviaBookingApi/types/IBookingVariantsResponse';
import {
    IBookingSegmentModel,
    IBookingVariantsModelType,
    IBookingVariantModel,
} from 'server/services/AviaBookingService/variants/types';
import {IReferenceDTO} from 'server/api/AviaBookingApi/types/IReferenceDTO';

import {flightDTOtoFlightModel} from 'server/utilities/lib/avia/booking/flightDTOtoFlightModel';

export class VariantsProcessError extends Error {
    constructor(message?: string) {
        const suffix = message ? `: ${message}` : '';

        super('Error while processing variants' + suffix);
    }
}

export function apiVariantsToVariantsBookingModel(
    variants: IBookingVariantsResponse,
): IBookingVariantsModelType {
    const legs = variants.variant_info.variant.legs;

    const passengers = variants.additional_data.passengers;

    const apiReference = variants.variant_info.reference;
    const segments = legs.map(leg => processFlights(leg.flights, apiReference));

    const priceInfo = variants.price_info;

    const allVariants = (variants.all_variants || []).map(x =>
        processVariant(x, apiReference),
    );

    const qid = variants.order_data.qid;

    const [forwardRoute, backwardRoute] = variants.variant.route;

    const flightsByRoute = {
        forward: getFlights(forwardRoute, variants.flights),
        backward: getFlights(backwardRoute, variants.flights),
    };

    const orderUrl = variants.order_data.url;

    return {
        passengers,
        priceInfo,
        segments,
        allVariants,
        qid,
        flightsByRoute,
        orderUrl,
    };
}

function getFlights(
    route: string[],
    flights: IBookingFlights,
): IBookingFlight[] {
    return route.map(flightId => flights[flightId]);
}

function processVariant(
    variant: IBookingAllVariant,
    reference: IReferenceDTO,
): IBookingVariantModel {
    return {
        id: variant.id,
        segments: variant.legs.map(leg =>
            processFlights(leg.flights, reference),
        ),
        priceInfo: variant.variantPriceInfo,
        promoCampaigns: variant.promoCampaigns,
        allowedLoyaltyPrograms: variant.allowedLoyaltyPrograms,
    };
}

function processFlights(
    rawFlights: IFlightDTO[],
    reference: IReferenceDTO,
): IBookingSegmentModel {
    return {
        flights: rawFlights.map(x => flightDTOtoFlightModel(x, reference)),
    };
}
