import moment from 'moment';

import {IAviaParams} from '../AviaSearchService/types/IAviaParams';

import {parseDate} from 'utilities/dateUtils';
import {
    IAviaBackendApiClient,
    IAviaBackendRequest,
} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';
import {ROBOT} from 'utilities/dateUtils/formats';

import {IDependencies} from 'server/getContainerConfig';

import {AviaGeoService} from '../AviaGeoService/AviaGeoService';

export interface ISchedule {
    hasRoutes: boolean;
}

export class AviaFlexibleService {
    private aviaGatewayBackendApiClient: IAviaBackendApiClient;

    private aviaGeoService: AviaGeoService;

    constructor({aviaGatewayBackendApiClient, aviaGeoService}: IDependencies) {
        this.aviaGatewayBackendApiClient = aviaGatewayBackendApiClient;
        this.aviaGeoService = aviaGeoService;
    }

    async getSchedule(params: IAviaParams): Promise<ISchedule | null> {
        const directionsData = await this.aviaGeoService.getDirectionsData(
            params,
        );

        if (!directionsData) {
            return null;
        }

        return this.aviaGatewayBackendApiClient.request<ISchedule>(
            this.getScheduleRequest(params),
        );
    }

    private getScheduleRequest(params: IAviaParams): IAviaBackendRequest {
        const today = moment();
        const when = parseDate(params.when);
        let returnDate;

        if (when) {
            returnDate = parseDate(params.return_date);
        }

        const [deltaBack, deltaNext] = getFlexDelta(today, when, returnDate);
        const leftDate = when.add(deltaBack, 'days').format(ROBOT);
        const rightDate = when.add(deltaNext, 'days').format(ROBOT);

        return {
            name: 'schedule',
            params: {
                fromPointKey: params.fromId,
                toPointKey: params.toId,
                leftDate,
                rightDate,
            },
            fields: ['hasRoutes'],
        };
    }
}

function getLastDayInCalendar(date: moment.Moment): moment.Moment {
    return date.add(1, 'years').date(0);
}

function getDeltaDays(d1: moment.Moment, d2: moment.Moment): number {
    return d1.diff(d2, 'days');
}

function getFlexDelta(
    today: moment.Moment,
    when: moment.Moment,
    returnDate?: moment.Moment,
): number[] {
    let deltaBack = -3;
    let deltaNext = 3;
    // Определение смещения дат на -/+ 3 дня от переданной

    const delta = getDeltaDays(when, today);

    if (delta < 3) {
        deltaBack += 3 - delta;
        deltaNext += 3 - delta;
    }

    const lastDay = getLastDayInCalendar(today);
    let lastDayDelta;

    if (returnDate) {
        lastDayDelta = getDeltaDays(lastDay, returnDate);
    } else {
        lastDayDelta = getDeltaDays(lastDay, when);
    }

    if (lastDayDelta < 3) {
        deltaBack -= 3 - lastDayDelta;
        deltaNext = lastDayDelta;
    }

    return [deltaBack, deltaNext];
}
