import {
    IFlightDetailedParams,
    IFlightInfo,
} from 'server/api/AviaFlightStorageApi/types/IFlight';
import {
    IAviaTDAnswer,
    IFlight,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';

import {
    isAirline,
    isFlight,
    isSettlement,
} from 'server/services/AviaFlightStorageService/utils/checks';
import {convertFlightToReference} from 'server/services/AviaFlightStorageService/utils/convertFlightToRefference';
import {CurrencyType} from 'utilities/currency/CurrencyType';

const convertToFlight = (
    params: IFlightDetailedParams[],
    flightsInfo: IFlightInfo[],
): IFlight[] =>
    params.reduce((result, flightParams, index) => {
        const convertedData = convertFlightToReference(
            flightsInfo,
            flightParams,
            index,
        );

        if (convertedData !== null) {
            result.push(convertedData);
        }

        return result;
    }, [] as IFlight[]);

export function convertFlightStorageData(
    forward: IFlightDetailedParams[],
    backward: IFlightDetailedParams[],
    flights: IFlightInfo[],
): IAviaTDAnswer | null {
    const flightsInfo = flights.filter(isFlight);

    // Если есть данные не по всем рейсам, то выходим.
    if (flightsInfo.length !== flights.length) {
        return null;
    }

    const airports = new Set([
        ...flightsInfo.map(f => f.airport_from),
        ...flightsInfo.map(f => f.airport_to),
    ]);
    const settlements = new Set(
        Array.from(airports.values())
            .map(station => station.settlement)
            .filter(isSettlement),
    );

    const forwardFlightsInfo = flightsInfo.slice(0, forward.length);
    const backwardFlightsInfo = flightsInfo.slice(forward.length);
    const forwardReference = convertToFlight(forward, forwardFlightsInfo);
    const backwardReference = convertToFlight(backward, backwardFlightsInfo);

    return {
        cont: 0,
        progress: {
            all: 0,
            current: 0,
        },
        variants: {
            fares: [
                {
                    route: [
                        forwardReference.map(({key}) => key),
                        backwardReference.map(({key}) => key),
                    ],
                    // Фейковая цена, которая исчезнет в верстке
                    prices: [
                        {
                            baggage: [['aviaFront'], ['aviaFront']],
                            partnerCode: 'aviaFront',
                            tariff_sign: 'aviaFront',
                            queryTime: 0,
                            tariff: {
                                value: 0,
                                currency: CurrencyType.RUB,
                            },
                            service: '',
                            boy: false,
                            fare_families_hash: 'aviaFront',
                            selfconnect: false,
                        },
                    ],
                },
            ],
        },
        reference: {
            alliances: [],
            aviaCompanies: [],
            baggageTariffs: {
                aviaFront: {},
            },
            companies: flightsInfo
                .filter(isFlight)
                .map(flight => flight.airline)
                .filter(isAirline)
                .map(airline => ({
                    id: airline.id,
                    logoSvg: airline.logoSvg,
                    title: airline.title,
                    url: '',
                    alliance: null,
                    color: airline.color,
                })),
            companyTariffs: [],
            flights: [...forwardReference, ...backwardReference],
            partners: [
                {
                    code: 'aviaFront',
                    id: 0,
                    title: '',
                },
            ],
            ratings: [],
            reviewsCount: {},
            settlements: Array.from(settlements).map(s => ({
                ...s,
                countryId: s.country.geoId,
                totalStations: 1,
            })),
            stations: Array.from(airports).map(s => ({
                ...s,
                tType: 'plane',
                settlement: s.settlement.geoId,
            })),
            fareFamilies: {},
        },
        partners: {
            aviaFront: 'done',
        },
    };
}
