import {
    IAviaDataByPointKey,
    IAviaCountryData,
} from 'server/services/AviaGeoService/types/IAviaDataByPointKey';
import {
    IAviaGeoServiceGetCountry,
    IAviaGeoService,
    IAviaGeoServiceGetCountries,
    IAviaGeoServiceGetDataByGeoId,
    IAviaGeoServiceGetDataByPointKey,
    IAviaGeoServiceGetDirectionsData,
} from 'server/services/AviaGeoService/types/IAviaGeoService';

import {IAviaBackendApiClient} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';
import {ILogger} from 'server/utilities/Logger';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';
import {unknownToErrorOrUndefined} from 'utilities/error';

import {IDependencies} from 'server/getContainerConfig';

export class AviaGeoService implements IAviaGeoService {
    private restApi: IAviaBackendApiClient;
    private logger: ILogger;

    constructor({aviaBackendRestApiClient, logger}: IDependencies) {
        this.restApi = aviaBackendRestApiClient;
        this.logger = logger;
    }

    /**
     * http://backend.testing.avia.yandex.net/rest/geo/country/205
     */
    getCountry: IAviaGeoServiceGetCountry = id => {
        return this.restApi.request<IAviaCountryData>({
            name: `geo/country/${id}`,
        });
    };

    /**
     * http://backend.testing.avia.yandex.net/rest/geo/country
     */
    getCountries: IAviaGeoServiceGetCountries = () => {
        return this.restApi.request<Record<number, IAviaCountryData>>({
            name: 'geo/country',
        });
    };

    /**
     * В качестве geoId скорее всего может выступать только geoId города,
     * но это не точно
     */
    getDataByGeoId: IAviaGeoServiceGetDataByGeoId = geoId => {
        return this.restApi.request<IAviaDataByPointKey>({
            name: 'geo/point',
            params: {geo_id: geoId},
        });
    };

    /**
     * http://backend.testing.avia.yandex.net/rest/geo/point?key=s9600370
     */
    getDataByPointKey: IAviaGeoServiceGetDataByPointKey = key => {
        return this.restApi.request<IAviaDataByPointKey>({
            name: 'geo/point',
            params: {key},
        });
    };

    getDirectionsData: IAviaGeoServiceGetDirectionsData = async params => {
        try {
            const {fromId, toId} = params;

            if (!isSettlementKey(fromId) && !isStationKey(fromId)) {
                throw new Error(
                    `From id '${fromId}' is not settlement or station point key`,
                );
            }

            if (!isSettlementKey(toId) && !isStationKey(toId)) {
                throw new Error(
                    `To id '${toId}' is not settlement or station point key`,
                );
            }

            const [from, to] = await Promise.all([
                this.getDataByPointKey(fromId),
                this.getDataByPointKey(toId),
            ]);

            if (!from) {
                throw new Error(`Didn't get point 'from' with key '${fromId}'`);
            }

            if (!to) {
                throw new Error(`Didn't get point 'to' with key '${toId}'`);
            }

            return {
                from,
                to,
            };
        } catch (e) {
            this.logger.logError(
                'AviaGeoService.getDirectionsData fail',
                unknownToErrorOrUndefined(e),
            );

            return null;
        }
    };
}
