import {
    INearDistances,
    INearDirection,
    INearItem,
    INearAnswer,
} from './types/IAviaNear';
import {IAviaParams} from 'server/services/AviaSearchService/types/IAviaParams';

import {
    greatCircleDistanceKm,
    ICoordinate,
} from 'server/utilities/earthGeometry';
import {
    IAviaBackendApiClient,
    IAviaBackendRequest,
} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';
import {ILogger} from 'server/utilities/Logger';
import {
    IAviaNearAirportsParams,
    packNearAirportsParams,
} from 'projects/avia/lib/packNearAirportsParams';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';

import {IDependencies} from 'server/getContainerConfig';
import {AviaGeoService} from 'server/services/AviaGeoService/AviaGeoService';

const defaultNearDistances = {
    defaultDistance: 100,
    minDistance: 50,
    maxDistance: 200,
};

export class AviaNearService {
    private api: IAviaBackendApiClient;
    private logger: ILogger;
    private aviaGeoService: AviaGeoService;

    constructor({
        aviaGatewayBackendApiClient,
        logger,
        aviaGeoService,
    }: IDependencies) {
        this.api = aviaGatewayBackendApiClient;
        this.aviaGeoService = aviaGeoService;
        this.logger = logger;
    }

    async getNearAirports(
        params: IAviaParams,
    ): Promise<INearAnswer | null | void> {
        try {
            const {fromId, toId} = params;

            if (!isSettlementKey(fromId) && !isStationKey(fromId)) {
                throw new Error(
                    `From id '${fromId}' is not settlement or station point key`,
                );
            }

            if (!isSettlementKey(toId) && !isStationKey(toId)) {
                throw new Error(
                    `To id '${toId}' is not settlement or station point key`,
                );
            }

            const [from, to] = await Promise.all([
                this.aviaGeoService.getDataByPointKey(fromId),
                this.aviaGeoService.getDataByPointKey(toId),
            ]);

            if (!to || !from) {
                this.logger.logError(
                    'getNearAirports',
                    new Error('Empty to/from points'),
                );

                return null;
            }

            const nearAirportsParams = packNearAirportsParams(params, from, to);
            const distance = await this.api.request<INearDistances>(
                this.getNearDistancesOptions(nearAirportsParams),
            );
            const nearDirections = await this.api.request<INearDirection[]>(
                this.getNearDirectionsOptions(
                    nearAirportsParams,
                    (distance || defaultNearDistances).maxDistance,
                ),
            );

            return {
                toPoint: {
                    name: nearAirportsParams.toName || '',
                    phraseFrom: nearAirportsParams.toPhraseFrom,
                    point: [
                        nearAirportsParams.toCoords.latitude,
                        nearAirportsParams.toCoords.longitude,
                    ],
                },
                nearPoints: (nearDirections || []).map(nearDirection =>
                    this.processNearItem(
                        nearAirportsParams.toCoords,
                        nearDirection,
                    ),
                ),
            };
        } catch (e) {}
    }

    private getNearDistancesOptions(
        params: IAviaNearAirportsParams,
    ): IAviaBackendRequest {
        return {
            name: 'nearDistances',
            params: {
                fromId: params.fromId,
                toId: params.toId,
            },
        };
    }

    private getNearDirectionsOptions(
        params: IAviaNearAirportsParams,
        distance: number,
    ): IAviaBackendRequest {
        return {
            name: 'nearDirections',
            params: {
                fromId: params.fromId,
                toId: params.toId,

                distance,

                when: params.when,
                returnDate: params.return_date || undefined,
                adultSeats: params.adult_seats,
                childrenSeats: params.children_seats,
                infantSeats: params.infant_seats,
                klass: params.klass,
            },
            fields: [
                {
                    name: 'fromCity',
                    fields: ['key', 'title', 'country'],
                },
                {
                    name: 'toCity',
                    fields: [
                        'key',
                        'title',
                        'geoId',
                        'country',
                        'latitude',
                        'longitude',
                        'iataCode',
                        'id',
                        'urlTitle',
                    ],
                },
                'price',
            ],
        };
    }

    private processNearItem(
        toRegion: ICoordinate,
        nearDirection: INearDirection,
    ): INearItem {
        const {fromCity, toCity, price} = nearDirection;

        const fromCountry = fromCity.country;
        const toCountry = toCity.country;

        let fullTitle;

        if (fromCountry && toCountry && fromCountry.id !== toCountry.id) {
            fullTitle = `${toCity.title}, ${toCountry.title}`;
        } else {
            fullTitle = toCity.title;
        }

        return {
            title: toCity.title,
            fullTitle,
            geoid: toCity.geoId,
            price,
            distance: greatCircleDistanceKm(toRegion, toCity),
            point: [toCity.latitude, toCity.longitude],
            iataCode: toCity.iataCode,
            id: toCity.id,
            urlTitle: toCity.urlTitle,
            country: toCity.country.title,
            key: toCity.key,
        };
    }
}
