import {
    IAviaGetPassengerExperienceParams,
    IPresumedFields,
    TAviaGetPassengerExperience,
} from './types/IAviaGetPassengerExperience';
import {IPassengerExperience} from 'projects/avia/types/IPassengerExperience';

import {ILogger} from 'server/utilities/Logger';

import {IDependencies} from 'server/getContainerConfig';
import {AviaPassengerExperienceApi} from 'server/api/AviaPassengerExperienceApi/AviaPassengerExperienceApi';

export interface IAviaPassengerExperienceService {
    getPassengerExperience: TAviaGetPassengerExperience;
}

export class AviaPassengerExperienceService
    implements IAviaPassengerExperienceService
{
    private logger: ILogger;
    private passengerExperienceApi: AviaPassengerExperienceApi;

    constructor({aviaPassengerExperienceApi, logger}: IDependencies) {
        this.passengerExperienceApi = aviaPassengerExperienceApi;
        this.logger = logger;
    }

    async getPassengerExperience(
        params: IAviaGetPassengerExperienceParams,
    ): Promise<IPassengerExperience | null> {
        const {planeNumber, when, klass, lang = 'ru'} = params;
        const [iata, flightNumber] = planeNumber.split(' ');

        if (!iata || !flightNumber || !when || !klass) {
            this.logger.logWarn('Недостаточно данных для обращения к API');

            return null;
        }

        const extras = await this.passengerExperienceApi.passengerExperience({
            iata,
            when,
            lang,
            flightNumber,
        });

        if (!extras) {
            return null;
        }

        const isEconomy = klass === 'economy';
        const planeModel =
            lang === 'ru' && extras.aircraft
                ? extras.aircraft.title
                : extras.title_en;

        const presumedFields: IPresumedFields = {
            ife: isEconomy ? extras.ife_economy : extras.ife_business,
            power: isEconomy ? extras.power_economy : extras.power_business,
            wifi: isEconomy ? extras.wifi_economy : extras.power_business,
            seatPitch: isEconomy
                ? extras.seat_pitch_economy
                : extras.seat_pitch_business,
            seatsTotal: extras.seats_total,
            planeModel,
            seatsInRow: extras.aircraft && extras.aircraft.plane_body_type,
            hasPropeller:
                extras.aircraft && extras.aircraft.is_propeller_flight,
        };

        const passengerExperience: IPassengerExperience = {};

        // Выбрасываем поля c null и undefined, для нас они не содержательны
        for (const key in presumedFields) {
            // eslint-disable-next-line no-eq-null
            if (presumedFields[key] != null) {
                passengerExperience[key] = presumedFields[key];
            }
        }

        return Object.keys(passengerExperience).length
            ? passengerExperience
            : null;
    }
}
