import {
    DEFAULT_ADULTS_COUNT,
    DEFAULT_INFANTS_COUNT,
    DEFAULT_CHILDREN_COUNT,
} from 'constants/avia';

import {IAviaCalendarPriceIndexService} from 'server/services/AviaPriceIndexService/types/IAviaCalendarPriceIndexService';
import {IAviaDynamicsPriceIndexService} from 'server/services/AviaPriceIndexService/types/IAviaDynamicsPriceIndexService';
import {IAviaMinPriceBatchSearchService} from 'server/services/AviaPriceIndexService/types/IAviaMinPriceBatchSearchService';

import convertCalendarPrices from './utilities/convertCalendarPrices';
import convertDynamicsPrices from './utilities/convertDynamicsPrices';
import getCalendarPriceMinRequests from './utilities/getCalendarPriceRequest/getCalendarPriceMinRequests';
import {isSettlementKey} from 'utilities/strings/isSettlementKey';
import {isStationKey} from 'utilities/strings/isStationKey';

import {IDependencies} from 'server/getContainerConfig';
import {AviaGeoService} from 'server/services/AviaGeoService/AviaGeoService';
import {AviaPriceIndexApi} from 'server/api/AviaPriceIndexApi/AviaPriceIndexApi';

export interface IAviaPriceIndexService {
    dynamics: IAviaDynamicsPriceIndexService;
    calendar: IAviaCalendarPriceIndexService;
    minPriceBatchSearch: IAviaMinPriceBatchSearchService;
}

export class AviaPriceIndexService implements IAviaPriceIndexService {
    private aviaPriceIndexApi: AviaPriceIndexApi;
    private aviaGeoService: AviaGeoService;

    constructor({aviaPriceIndexApi, aviaGeoService}: IDependencies) {
        this.aviaPriceIndexApi = aviaPriceIndexApi;
        this.aviaGeoService = aviaGeoService;
    }

    dynamics: IAviaDynamicsPriceIndexService = async (
        interval,
        params,
        filters,
    ) => {
        const dynamicsPrices = await this.aviaPriceIndexApi.search(
            interval,
            params,
            filters,
        );

        return convertDynamicsPrices(dynamicsPrices);
    };

    calendar: IAviaCalendarPriceIndexService = async ({
        fromId,
        toId,
        adultsCount = DEFAULT_ADULTS_COUNT,
        childrenCount = DEFAULT_CHILDREN_COUNT,
        infantsCount = DEFAULT_INFANTS_COUNT,
        forwardDate,
        backwardDate,
    }) => {
        if (!isSettlementKey(fromId) && !isStationKey(fromId)) {
            return null;
        }

        if (!isSettlementKey(toId) && !isStationKey(toId)) {
            return null;
        }

        const [from, to] = await Promise.all([
            this.aviaGeoService.getDataByPointKey(fromId),
            this.aviaGeoService.getDataByPointKey(toId),
        ]);

        if (!from?.settlement || !to?.settlement) {
            return null;
        }

        const minRequests = getCalendarPriceMinRequests({
            fromId: from.settlement.id,
            toId: to.settlement.id,
            adultsCount,
            childrenCount,
            infantsCount,
            forwardDate,
            backwardDate: backwardDate || null,
        });

        const response = await this.aviaPriceIndexApi.minPriceBatchSearch({
            minRequests,
        });

        return response?.data
            ? convertCalendarPrices(response.data, Boolean(forwardDate))
            : {};
    };

    minPriceBatchSearch: IAviaMinPriceBatchSearchService = async params =>
        this.aviaPriceIndexApi.minPriceBatchSearch(params);
}
