import {flatten, get} from 'lodash';
import * as crypto from 'crypto';

import {
    DEFAULT_ADULTS_COUNT,
    DEFAULT_AVIA_CLASS,
    DEFAULT_INFANTS_COUNT,
    DEFAULT_CHILDREN_COUNT,
} from 'constants/avia';

import {
    IAviaRedirectPageParams,
    IAviaTDRedirectAnswer,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDRedirect';
import {
    IAviaTDAnswer,
    IFare,
    IVariantPrice,
} from 'server/api/AviaTicketDaemonApi/types/IAviaTDAnswer';
import {IUtmQueryParams} from 'types/common/IQueryParams';
import {IAviaParams} from '../AviaSearchService/types/IAviaParams';
import {IAviaRedirectVariantParams} from './types/IAviaRedirectVariantParams';
import {IAviaRedirectWizardParams} from './types/IAviaRedirectWizardParams';
import {IAviaTDRedirectSettings} from './types/IAviaTDRedirectSettings';
import {IAviaRedirectVariantStatus} from './types/IAviaRedirectVariantStatus';
import {IAviaTDRedirectQuery} from 'server/api/AviaTicketDaemonApi/types/IAviaTDRedirectQuery';

import {
    INormalizedTDReference,
    normalizeTDReference,
} from 'reducers/avia/utils/ticketDaemon/normalizeTDReference';

import {
    denormalizePartnersBaggage,
    parseAviaBaggage,
} from 'selectors/avia/utils/denormalization/baggage';

import {ILogger} from 'server/utilities/Logger';
import {decodeQidOrDefault} from 'projects/avia/lib/qid';
import {isBaggageIncluded} from 'projects/avia/lib/baggage';
import {unknownToErrorOrUndefined} from 'utilities/error';

import {IDependencies} from 'server/getContainerConfig';
import {AviaTicketDaemonApi} from 'server/api/AviaTicketDaemonApi/AviaTicketDaemonApi';
import {TestContextService} from 'server/services/TestContextService/TestContextService';

import {AviaVariantsService} from '../AviaVariantsService/AviaVariantsService';

export class AviaRedirectService {
    private keySecret = 'PrKbleSBHhLqvfmWxYRsZnpzaJFcQGVu';

    private aviaTicketDaemonApi: AviaTicketDaemonApi;
    private aviaVariantsService: AviaVariantsService;
    private testContextService: TestContextService;

    private logger: ILogger;

    constructor({
        aviaTicketDaemonApi,
        aviaVariantsService,
        testContextService,
        logger,
    }: IDependencies) {
        this.aviaTicketDaemonApi = aviaTicketDaemonApi;
        this.aviaVariantsService = aviaVariantsService;
        this.testContextService = testContextService;
        this.logger = logger;
    }

    /**
     * На странице редиректа происходит перепоиск варианта, контроллер прокидывает вызов сюда.
     * Для редиректа к партнеру клиент ждет отсюда ответ {found: true}, иначе ретраим или переходим на
     * ошибку редиректа. Состояние ретраев хранится на клиенте.
     *
     * Если редирект без багажа, то ждем первых вариантов от тд и отправляем {found: true}. Если с багажом,
     * то проверяем наличие багажа в полученных вариантах и соотвествующе выставляем признак found.
     * Если тд не отдал варианты, то проверяем статус опроса партнера. Если его опрос закончен, то выставляем
     * флаг {complete: true}. Это наследие легаси, суть в том, что партнер уже отдал данные, но они не дореплицировались
     * до дц, в который ушел запрос. В этом случае на клиенте будет сброшенно количество ретраев до минимума и продолжится
     * поллинг в ожидании репликации (но тут нет гарантии, что в данных что-то будет).
     * Возникающие ошибки здесь приводят просто к декременту ретраев на клиенте.
     */
    async checkVariantExistence(
        params: IAviaRedirectPageParams,
        nationalVersionForPartnersDisabling: Nullable<string>,
    ): Promise<IAviaRedirectVariantStatus> {
        const qidParams = decodeQidOrDefault(params.qid);

        const data = await this.aviaVariantsService.variants(
            {
                qid: params.qid,
                adults: `${qidParams.adult_seats || DEFAULT_ADULTS_COUNT}`,
                children: `${
                    qidParams.children_seats || DEFAULT_CHILDREN_COUNT
                }`,
                infants: `${qidParams.infant_seats || DEFAULT_INFANTS_COUNT}`,
                klass: qidParams.klass || DEFAULT_AVIA_CLASS,
                partners: params.partner,
                forward: params.forward,
                point_from: qidParams.fromId,
                point_to: qidParams.toId,
                date_forward: qidParams.when,
                backward: params.backward || undefined,
                date_backward: qidParams.return_date || undefined,
            },
            nationalVersionForPartnersDisabling,
        );

        if (!data || !params.partner) {
            throw new Error('No order data');
        }

        const fares = get(data, ['variants', 'fares'], []);

        const status = {
            complete: this.isPartnerDone(data, params.partner),
            found: Boolean(fares.length),
        };

        const hasBaggage = params.withBaggage === '1';

        if (hasBaggage && fares.length) {
            const partnerVariants = this.getPartnerVariants(
                fares[0],
                params.partner,
            );

            status.found = this.hasVariantsBaggage(
                partnerVariants,
                normalizeTDReference(data.reference, data.partners || {}),
            );
        }

        return status;
    }

    async getRedirectData(
        searchForm: IAviaParams,
        variantParams: IAviaRedirectVariantParams,
        wizardParams: IAviaRedirectWizardParams,
        utms: IUtmQueryParams,
        redirectSettings: IAviaTDRedirectSettings,
    ): Promise<{data: {data: IAviaTDRedirectAnswer}; status: number}> {
        const wizardQuery = await this.getWizardQueryParams(wizardParams);

        const queryParams: IAviaTDRedirectQuery = {
            ...wizardQuery,
            ...this.getSearchQueryParams(searchForm),
            ...this.getVariantQueryParams(variantParams),
            ...redirectSettings,
            ...utms,
            variantTestContext:
                await this.testContextService.getAviaTestContextTokenIfNeeded(
                    redirectSettings.variantTestContext,
                ),
        };

        return this.aviaTicketDaemonApi.redirect(queryParams);
    }

    private getPartnerVariants(
        {prices}: IFare,
        partner: string,
    ): IVariantPrice[] {
        return prices.filter(({partnerCode}) => partnerCode === partner);
    }

    private hasVariantsBaggage(
        variants: IVariantPrice[],
        reference: INormalizedTDReference,
    ): boolean {
        return variants.some(price => {
            const denormalizedBaggage = denormalizePartnersBaggage(
                price,
                reference.baggageTariffs,
            );

            const baggage = parseAviaBaggage(flatten(denormalizedBaggage));

            return isBaggageIncluded(baggage);
        });
    }

    private isPartnerDone(
        {partners}: IAviaTDAnswer,
        partnerCode: string,
    ): boolean {
        return partners?.[partnerCode] === 'done';
    }

    private async getWizardQueryParams(
        wizardParams: IAviaRedirectWizardParams,
    ): Promise<IAviaRedirectWizardParams> {
        const {wizardTariffKey, ...rest} = wizardParams;
        let wizardPrice = {};

        try {
            if (wizardTariffKey) {
                const [price, sign] = wizardTariffKey.split('|');

                if (price && sign) {
                    const isValid = await this.validateTariffKey(sign, price);

                    if (isValid) {
                        const priceMatch = price.match(/^(\d+)(\w{3})$/);

                        wizardPrice =
                            priceMatch === null
                                ? {}
                                : {
                                      revise_price_value: priceMatch[1],
                                      revise_price_currency: priceMatch[2],
                                  };
                    }
                }
            }
        } catch (e) {
            this.logger.logWarn('redirectData', unknownToErrorOrUndefined(e));
        }

        return {
            ...wizardPrice,
            ...rest,
        };
    }

    private validateTariffKey(key: string, valid: string): Promise<unknown> {
        const hmac = crypto.createHmac('md5', this.keySecret);

        return new Promise(resolve => {
            hmac.on('readable', () => {
                const data = hmac.read() as Buffer;

                resolve(data && data.toString('hex') === key);
            });

            hmac.write(valid);
            hmac.end();
        });
    }

    private getSearchQueryParams(
        searchForm: IAviaParams,
    ): Pick<
        IAviaTDRedirectQuery,
        | 'point_from'
        | 'point_to'
        | 'date_forward'
        | 'date_backward'
        | 'klass'
        | 'adults'
        | 'children'
        | 'infants'
    > {
        return {
            point_from: searchForm.fromId,
            point_to: searchForm.toId,
            date_forward: searchForm.when,
            date_backward: searchForm.return_date || undefined,
            klass: searchForm.klass,
            adults: searchForm.adult_seats,
            children: searchForm.children_seats,
            infants: searchForm.infant_seats,
        };
    }

    private getVariantQueryParams(
        variantParams: IAviaRedirectVariantParams,
    ): Pick<
        IAviaTDRedirectQuery,
        | 'qid'
        | 'partner'
        | 'forward'
        | 'backward'
        | 'with_baggage'
        | 'tariff_sign'
        | 'fare_families_hash'
        | 'avia_brand'
    > {
        return {
            qid: variantParams.qid,
            partner: variantParams.partner,
            forward: variantParams.forward,
            backward: variantParams.backward || undefined,
            with_baggage: Number(variantParams.withBaggage) === 1 || undefined,
            tariff_sign: variantParams.tariff_sign,
            fare_families_hash: variantParams.fare_families_hash,
            avia_brand: variantParams.avia_brand,
        };
    }
}
