import moment from 'moment';

import {
    IAviaTopFlight,
    IAviaTopFlights,
    IAviaTopFlightsParams,
} from 'server/services/AviaSearchService/types/IAviaTopFlights';
import {IAviaInitSearchService} from 'server/services/AviaSearchService/types/IAviaInitSearchService';
import {ICookies} from '@yandex-data-ui/core/lib/types';

import {ROBOT} from 'utilities/dateUtils/formats';
import {ILogger} from 'server/utilities/Logger';
import {parseAviaDate} from 'projects/avia/lib/date/parseDate';
import {
    IAviaBackendApiClient,
    IAviaBackendRequest,
} from 'server/utilities/AviaBackendApiClient/IAviaBackendApiClient';
import {unknownToErrorOrUndefined} from 'utilities/error';
import {getAviaTestContext} from 'server/utilities/avia/getAviaTestContext';

import {IDependencies} from 'server/getContainerConfig';
import {AviaTicketDaemonApi} from 'server/api/AviaTicketDaemonApi/AviaTicketDaemonApi';
import {
    AviaPartnersApi,
    IPartner,
    IPartnerInfo,
} from 'server/api/AviaPartnersApi/AviaPartnersApi';

export class AviaSearchService {
    private aviaTicketDaemonApi: AviaTicketDaemonApi;
    private aviaPartnersApi: AviaPartnersApi;
    private aviaGatewayBackendApiClient: IAviaBackendApiClient;
    private logger: ILogger;
    private readonly cookies: ICookies;

    constructor({
        aviaGatewayBackendApiClient,
        aviaTicketDaemonApi,
        aviaPartnersApi,
        logger,
        cookies,
    }: IDependencies) {
        this.aviaTicketDaemonApi = aviaTicketDaemonApi;
        this.aviaGatewayBackendApiClient = aviaGatewayBackendApiClient;
        this.aviaPartnersApi = aviaPartnersApi;
        this.logger = logger;
        this.cookies = cookies;
    }

    initSearch: IAviaInitSearchService = async searchParams => {
        return this.aviaTicketDaemonApi.initByQuery({
            adults: searchParams.adult_seats,
            children: searchParams.children_seats,
            infants: searchParams.infant_seats,
            date_backward: parseAviaDate(searchParams.return_date),
            date_forward: parseAviaDate(searchParams.when),
            klass: searchParams.klass,
            lang: 'ru',
            national: 'ru',
            point_from: searchParams.fromId,
            point_to: searchParams.toId,
            test_context: getAviaTestContext({
                cookies: this.cookies,
            }),
        });
    };

    async getPopularPartners(
        fromPointKey: string,
        toPointKey: string,
    ): Promise<string[]> {
        const MAX_SCORE = 0.6;

        try {
            const partners = await this.aviaPartnersApi.getPopularPartners(
                fromPointKey,
                toPointKey,
            );

            if (!partners) {
                return [];
            }

            return partners.reduce(
                (acc, partner) => {
                    if (acc.score > MAX_SCORE) {
                        return acc;
                    }

                    acc.partners.push(partner.partner_code);
                    acc.score += partner.score;

                    return acc;
                },
                // eslint-disable-next-line no-array-constructor
                {partners: Array<string>(), score: 0},
            ).partners;
        } catch (ex) {
            this.logger.logError(
                'AviaSearchService.getPopularPartners failure',
                unknownToErrorOrUndefined(ex),
            );

            return [];
        }
    }

    getPartner(partner: string): Promise<IPartnerInfo | null> {
        return this.aviaPartnersApi.getPartner(partner);
    }

    async topFlights({
        fromId,
        toId,
        when,
        returnDate,
    }: IAviaTopFlightsParams): Promise<IAviaTopFlights> {
        const getTopFlightsRequestParams = (
            date: string,
        ): IAviaBackendRequest => ({
            name: 'topFlights',
            params: {
                fromKey: fromId,
                toKey: toId,
                date,
                limit: 100,
            },
        });

        const dateForward = moment(when, ROBOT);
        const dateBackward = returnDate && moment(returnDate, ROBOT);

        try {
            const forward =
                (await this.aviaGatewayBackendApiClient.request<
                    IAviaTopFlight[]
                >(getTopFlightsRequestParams(dateForward.format(ROBOT)))) || [];
            const backward =
                (dateBackward &&
                    (await this.aviaGatewayBackendApiClient.request<
                        IAviaTopFlight[]
                    >(
                        getTopFlightsRequestParams(dateBackward.format(ROBOT)),
                    ))) ||
                [];

            return {
                forward: forward.map(item => item.numbers),
                backward: backward.map(item => item.numbers),
                forward2: forward,
                backward2: backward,
            };
        } catch (e) {
            return {
                forward: [],
                backward: [],
                forward2: [],
                backward2: [],
            };
        }
    }

    async getPartnersInfo(): Promise<Record<string, IPartner>> {
        const partnersMap = await this.aviaPartnersApi.getAll();
        const partners: Record<string, IPartner> = {};

        partnersMap.forEach((value, key) => {
            partners[key] = value;
        });

        return partners;
    }
}
